/* Ergo, version 3.8, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2019 Elias Rudberg, Emanuel H. Rubensson, Pawel Salek,
 * and Anastasia Kruchinina.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Ergo: An open-source program for linear-scaling electronic structure
 * calculations,
 * Elias Rudberg, Emanuel H. Rubensson, Pawel Salek, and Anastasia
 * Kruchinina,
 * SoftwareX 7, 107 (2018),
 * <http://dx.doi.org/10.1016/j.softx.2018.03.005>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

/*-*-mode: C; c-indentation-style: "bsd"; c-basic-offset: 4; -*-*/
/** @file fun-pw91x.c  PW91X implementation.

   Automatically generated code implementing PW91X functional and
   its derivatives. It is generated by func-codegen.pl being a part of
   a "Automatic code generation framework for analytical functional
   derivative evaluation", Pawel Salek, 2005

    This functional is connected by making following changes:
    1. add "extern Functional pw91xFunctional;" to 'functionals.h'
    2. add "&pw91xFunctional," to 'functionals.c'
    3. add "fun-pw91x.c" to 'Makefile.am', 'Makefile.in' or 'Makefile'.

    This functional has been generated from following input:
    ------ cut here -------
 This is exchange PW91x functional of 

J.P. Perdew, in Proceedings of the 21st Annual International Symposium
on the Electronic Structure of Solids, edited by P. Ziesche and
H. Eschrig (akademie Verlag, Berlin 1991).


Cx:    -3/4*(6/%PI)^(1/3);
alpha: -3/2*(3/(4*%PI))^(1/3);
b:    0.0042;
c:    1.6455;
d:    4;
bet:  5*((36*%PI)^(-5/3));

x(r,g) := g*r^(-4/3);
F0(x):=b*x^2-(b-bet)*x^2*exp(-c*x^2)-10^(-6)*x^d;
F1(x):=1+6*b*x*asinh(x)-(10^(-6)*x^d)/(alpha);

F91(x) := alpha - F0(x)/F1(x);

Fb88(x) := alpha - b*x^2/(1 + 6*b*x*asinh(x));

Fd(x) := alpha;

Ex(r,g) := r^(4/3)*F91( x(r,g) );

K(rhoa,grada,rhob,gradb,gradab):= Ex(rhoa,grada) +Ex(rhob,gradb);

    ------ cut here -------
*/

#include <math.h>
#include <stddef.h>
 
#define __CVERSION__
 
#include "functionals.h"
 
/* INTERFACE PART */
static int pw91x_isgga(void) { return 1; } /* FIXME: detect! */
static int pw91x_read(const char *conf_line);
static real pw91x_energy(const FunDensProp* dp);
static void pw91x_first(FunFirstFuncDrv *ds,   real factor,
                         const FunDensProp* dp);
static void pw91x_second(FunSecondFuncDrv *ds, real factor,
                          const FunDensProp* dp);
static void pw91x_third(FunThirdFuncDrv *ds,   real factor,
                         const FunDensProp* dp);
static void pw91x_fourth(FunFourthFuncDrv *ds,   real factor,
                          const FunDensProp* dp);
 
Functional Pw91xFunctional = {
  "PW91X",       /* name */
  pw91x_isgga,   /* gga-corrected */
  pw91x_read,
  NULL,
  pw91x_energy,
  pw91x_first,
  pw91x_second,
  pw91x_third,
  pw91x_fourth
};
 
/* IMPLEMENTATION PART */
static int
pw91x_read(const char *conf_line)
{
    fun_set_hf_weight(0);
    return 1;
}



static real
pw91x_energy(const FunDensProp *dp)
{
    real res;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb;

    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;

    t1 = POW(rhoa,1.333333333333333);
    t2 = POW(3.0,.3333333333333333);
    t3 = POW(4.0,.3333333333333333);
    t4 = POW(M_PI,.3333333333333333);
    t5 = -1.5*t2/(t3*t4);
    t6 = 1/t2;
    t7 = POW(grada,4.0);
    t8 = 1/POW(rhoa,5.333333333333333);
    t9 = 1/t1;
    t10 = POW(grada,2.0);
    t11 = 1/POW(rhoa,2.666666666666667);
    t12 = 0.0042-.1388888888888889/(POW(36.0,.6666666666666666)*
        POW(M_PI,1.666666666666667));
    t13 = POW(rhob,1.333333333333333);
    t14 = POW(gradb,4.0);
    t15 = 1/POW(rhob,5.333333333333333);
    t16 = 1/t13;
    t17 = POW(gradb,2.0);
    t18 = 1/POW(rhob,2.666666666666667);

   /* code */
    res = t1*(t5-1.0*(-9.999999999999999e-7*t7*t8-1.0*t10*
        t11*t12/POW(2.718281828459045,1.6455*t10*t11)+0.0042*t10*t11)/
        (0.0252*grada*ASINH(grada*t9)*t9+6.666666666666666e-7*t3*t4*
        t6*t7*t8+1.0))+t13*(t5-1.0*(-1.0*t12*t17*t18/POW(2.718281828459045,
        1.6455*t17*t18)+0.0042*t17*t18-9.999999999999999e-7*t14*t15)/
        (6.666666666666666e-7*t14*t15*t3*t4*t6+0.0252*gradb*ASINH(gradb*
        t16)*t16+1.0));

    /* ELIAS NOTE 2016-07-14: for single precsion, the returned res
       value here can be nan or inf; similar problems seem to exist
       for several other functionals also. So, this can only be
       expected to work when at least double precision is used. The
       xcmat_nan_inf_test.cc test was changed so that some functionals
       are skipped in the single-precision case. */

    return res;
}

static void
pw91x_first_helper(real rhoa, real grada, real *res)
{    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21;

    t1 = POW(rhoa,1.333333333333333);
    t2 = POW(3.0,.3333333333333333);
    t3 = POW(4.0,.3333333333333333);
    t4 = POW(M_PI,.3333333333333333);
    t5 = POW(grada,4.0);
    t6 = 1/POW(rhoa,5.333333333333333);
    t7 = 1/t1;
    t8 = ASINH(grada*t7);
    t9 = 0.0252*grada*t8*t7+6.666666666666666e-7*t3*t4*t5*
        t6/t2+1.0;
    t10 = 1/t9;
    t11 = 1/POW(rhoa,6.333333333333333);
    t12 = POW(grada,2.0);
    t13 = 1/POW(rhoa,3.666666666666667);
    t14 = 0.0042-.1388888888888889/(POW(36.0,.6666666666666666)*
        POW(M_PI,1.666666666666667));
    t15 = 1/POW(rhoa,2.666666666666667);
    t16 = 1/POW(2.718281828459045,1.6455*t12*t15);
    t17 = 1.9999999999999998e-6/POW(3.0,1.333333333333333);
    t18 = 1/
        SQRT(t12*t15+1.0);
    t19 = 1/POW(t9,2.0);
    t20 = -9.999999999999999e-7*t5*t6-1.0*t12*t14*t15*t16+
        0.0042*t12*t15;
    t21 = POW(grada,3.0);

   /* code */
    res[0] = t1*(t19*t20*(-0.0336*t8*grada/POW(rhoa,2.333333333333333)-
        5.333333333333333*t11*t17*t3*t4*t5-0.0336*t12*t18*t13)-1.0*
        t10*(5.333333333333333e-6*t11*t5+2.666666666666667*t12*t13*
        t14*t16-4.388*t14*t5*t11*t16-0.0112*t12*t13))+1.333333333333333*
        (-1.5*t2/(t3*t4)-1.0*t10*t20)*POW(rhoa,.3333333333333333);
    res[1] = t1*(t19*t20*(0.0252*t8*t7+4.0*t17*t21*t3*t4*
        t6+0.0252*grada*t18*t15)-1.0*t10*(-2.0*t14*t15*t16*grada-3.9999999999999997e-6*
        t21*t6+3.291*t14*t21*t6*t16+0.0084*grada*t15));
}

static void
pw91x_first(FunFirstFuncDrv *ds, real factor, const FunDensProp *dp)
{
    real res[2];

    pw91x_first_helper(dp->rhoa, dp->grada, res);
   /* Final assignment */
    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];


    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pw91x_first_helper(dp->rhob, dp->gradb, res);
    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];

}

static void
pw91x_second_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36;

    t1 = POW(rhoa,1.333333333333333);
    t2 = POW(3.0,.3333333333333333);
    t3 = POW(4.0,.3333333333333333);
    t4 = POW(M_PI,.3333333333333333);
    t5 = POW(grada,4.0);
    t6 = 1/POW(rhoa,5.333333333333333);
    t7 = 1/t1;
    t8 = ASINH(grada*t7);
    t9 = 0.0252*grada*t8*t7+6.666666666666666e-7*t3*t4*t5*
        t6/t2+1.0;
    t10 = 1/t9;
    t11 = 1/POW(rhoa,6.333333333333333);
    t12 = POW(grada,2.0);
    t13 = 1/POW(rhoa,3.666666666666667);
    t14 = 0.0042-.1388888888888889/(POW(36.0,.6666666666666666)*
        POW(M_PI,1.666666666666667));
    t15 = 1/POW(rhoa,2.666666666666667);
    t16 = 1/POW(2.718281828459045,1.6455*t12*t15);
    t17 = 5.333333333333333e-6*t11*t5+2.666666666666667*t12*
        t13*t14*t16-4.388*t14*t5*t11*t16-0.0112*t12*t13;
    t18 = 1/POW(3.0,1.333333333333333);
    t19 = 1.9999999999999998e-6*t18;
    t20 = SQRT(t12*t15+1.0);
    t21 = 1/t20;
    t22 = 1/POW(rhoa,2.333333333333333);
    t23 = -5.333333333333333*t11*t19*t3*t4*t5-0.0336*grada*
        t8*t22-0.0336*t12*t21*t13;
    t24 = 1/POW(t9,2.0);
    t25 = -9.999999999999999e-7*t5*t6-1.0*t12*t14*t15*t16+
        0.0042*t12*t15;
    t26 = t23*t24*t25-1.0*t10*t17;
    t27 = POW(rhoa,.3333333333333333);
    t28 = -1.5*t2/(t3*t4)-1.0*t10*t25;
    t29 = POW(grada,3.0);
    t30 = -2.0*t14*t15*t16*grada-3.9999999999999997e-6*t29*
        t6+3.291*t14*t29*t6*t16+0.0084*grada*t15;
    t31 = 0.0252*t8*t7+4.0*t19*t29*t3*t4*t6+0.0252*grada*
        t21*t15;
    t32 = t31*t24*t25-1.0*t10*t30;
    t33 = 1/POW(rhoa,7.333333333333333);
    t34 = 1/POW(rhoa,4.666666666666667);
    t35 = 1/POW(t9,3.0);
    t36 = 1/POW(t20,3.0);

   /* code */
    res[0] = 1.333333333333333*t27*t28+t1*t26;
    res[1] = t1*t32;
    res[2] = t1*(-1.0*t10*(-19.254544*t14*t16*POW(grada,6.0)/
        POW(rhoa,10.0)-3.377777777777778e-5*t33*t5-9.777777777777779*
        t12*t14*t16*t34+.04106666666666666*t12*t34+39.492*t14*t5*t33*
        t16)+t24*t25*(0.0784*t8*grada/POW(rhoa,3.333333333333333)+
        6.755555555555556e-5*t18*t3*t33*t4*t5+0.168*t12*t21*t34-.04479999999999999*
        t5*t36*t33)-2.0*POW(t23,2.0)*t25*t35+2.0*t17*t23*t24)+.4444444444444444*
        t28/POW(rhoa,.6666666666666666)+2.666666666666667*t26*t27;
    res[3] = t1*(-1.0*t10*(14.440908*t14*t16*POW(grada,5.0)/
        POW(rhoa,9.0)+5.333333333333333*t13*t14*t16*grada+2.1333333333333332e-5*
        t11*t29-26.328*t14*t29*t11*t16-0.0224*grada*t13)+t24*t25*(-
        4.2666666666666664e-5*t11*t18*t29*t3*t4-0.0336*t8*t22-0.1008*
        grada*t21*t13+0.0336*t29*t36*t11)-2.0*t23*t25*t31*t35+t23*
        t24*t30+t31*t24*t17)+1.333333333333333*t27*t32;
    res[4] = t1*(-1.0*t10*(-10.830681*t14*t16*t5/POW(rhoa,
        8.0)-1.2e-5*t12*t6+16.455*t14*t12*t6*t16-2.0*t14*t15*t16+0.0084*
        t15)+t24*t25*(6.e-6*POW(4.0,1.333333333333333)*t12*t18*t4*
        t6-0.0252*t12*t36*t6+0.0504*t21*t15)-2.0*t25*POW(t31,2.0)*
        t35+2.0*t24*t30*t31);

}

static void
pw91x_second(FunSecondFuncDrv *ds, real factor, const FunDensProp* dp)
{
    real res[5];
 
    pw91x_second_helper(dp->rhoa, dp->grada, res);

    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];

    ds->df2000 += factor*res[2];
    ds->df1010 += factor*res[3];
    ds->df0020 += factor*res[4];


    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pw91x_second_helper(dp->rhob, dp->gradb, res);
    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];

    ds->df0200 += factor*res[2];
    ds->df0101 += factor*res[3];
    ds->df0002 += factor*res[4];

}

static void
pw91x_third_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36, t37, t38, t39, t40, t41, t42;
    real t43, t44, t45, t46, t47, t48, t49, t50;
    real t51, t52, t53, t54, t55, t56, t57, t58;
    real t59, t60;

    t1 = POW(rhoa,1.333333333333333);
    t2 = POW(3.0,.3333333333333333);
    t3 = POW(4.0,.3333333333333333);
    t4 = POW(M_PI,.3333333333333333);
    t5 = POW(grada,4.0);
    t6 = 1/POW(rhoa,5.333333333333333);
    t7 = 1/t1;
    t8 = ASINH(grada*t7);
    t9 = 0.0252*grada*t8*t7+6.666666666666666e-7*t3*t4*t5*
        t6/t2+1.0;
    t10 = 1/t9;
    t11 = 1/POW(rhoa,6.333333333333333);
    t12 = POW(grada,2.0);
    t13 = 1/POW(rhoa,3.666666666666667);
    t14 = 0.0042-.1388888888888889/(POW(36.0,.6666666666666666)*
        POW(M_PI,1.666666666666667));
    t15 = 1/POW(rhoa,2.666666666666667);
    t16 = 1/POW(2.718281828459045,1.6455*t12*t15);
    t17 = 5.333333333333333e-6*t11*t5+2.666666666666667*t12*
        t13*t14*t16-4.388*t14*t5*t11*t16-0.0112*t12*t13;
    t18 = 1/POW(3.0,1.333333333333333);
    t19 = 1.9999999999999998e-6*t18;
    t20 = SQRT(t12*t15+1.0);
    t21 = 1/t20;
    t22 = 1/POW(rhoa,2.333333333333333);
    t23 = -5.333333333333333*t11*t19*t3*t4*t5-0.0336*grada*
        t8*t22-0.0336*t12*t21*t13;
    t24 = 1/POW(t9,2.0);
    t25 = -9.999999999999999e-7*t5*t6-1.0*t12*t14*t15*t16+
        0.0042*t12*t15;
    t26 = t23*t24*t25-1.0*t10*t17;
    t27 = POW(rhoa,.3333333333333333);
    t28 = -1.5*t2/(t3*t4)-1.0*t10*t25;
    t29 = POW(grada,3.0);
    t30 = -2.0*t14*t15*t16*grada-3.9999999999999997e-6*t29*
        t6+3.291*t14*t29*t6*t16+0.0084*grada*t15;
    t31 = 0.0252*t8*t7+4.0*t19*t29*t3*t4*t6+0.0252*grada*
        t21*t15;
    t32 = t31*t24*t25-1.0*t10*t30;
    t33 = 1/POW(rhoa,7.333333333333333);
    t34 = 1/POW(rhoa,4.666666666666667);
    t35 = POW(grada,6.0);
    t36 = 1/POW(rhoa,10.0);
    t37 = -3.377777777777778e-5*t33*t5-9.777777777777779*
        t12*t14*t16*t34+.04106666666666666*t12*t34-19.254544*t14*t35*
        t36*t16+39.492*t14*t5*t33*t16;
    t38 = POW(t23,2.0);
    t39 = 1/POW(t9,3.0);
    t40 = 1/POW(t20,3.0);
    t41 = 1/POW(rhoa,3.333333333333333);
    t42 = 6.755555555555556e-5*t18*t3*t33*t4*t5+0.0784*grada*
        t8*t41+0.168*t12*t21*t34-.04479999999999999*t5*t40*t33;
    t43 = -2.0*t25*t38*t39-1.0*t10*t37+t42*t24*t25+2.0*t17*
        t23*t24;
    t44 = 1/POW(rhoa,.6666666666666666);
    t45 = POW(grada,5.0);
    t46 = 1/POW(rhoa,9.0);
    t47 = 5.333333333333333*t13*t14*t16*grada+2.1333333333333332e-5*
        t11*t29+14.440908*t14*t45*t46*t16-26.328*t14*t29*t11*t16-0.0224*
        grada*t13;
    t48 = -4.2666666666666664e-5*t11*t18*t29*t3*t4-0.0336*
        t8*t22-0.1008*grada*t21*t13+0.0336*t29*t40*t11;
    t49 = -1.0*t10*t47-2.0*t23*t25*t31*t39+t23*t24*t30+t48*
        t24*t25+t31*t24*t17;
    t50 = 1/POW(rhoa,8.0);
    t51 = -1.2e-5*t12*t6+16.455*t14*t12*t6*t16-10.830681*
        t14*t5*t50*t16-2.0*t14*t15*t16+0.0084*t15;
    t52 = POW(t31,2.0);
    t53 = POW(4.0,1.333333333333333);
    t54 = 6.e-6*t12*t18*t4*t53*t6-0.0252*t12*t40*t6+0.0504*
        t21*t15;
    t55 = -2.0*t25*t39*t52-1.0*t10*t51+2.0*t24*t30*t31+t54*
        t24*t25;
    t56 = 1/POW(rhoa,8.333333333333334);
    t57 = 1/POW(rhoa,5.666666666666667);
    t58 = 1/POW(rhoa,11.0);
    t59 = 1/POW(t9,4.0);
    t60 = 1/POW(t20,5.0);

   /* code */
    res[0] = 1.333333333333333*t27*t28+t1*t26;
    res[1] = t1*t32;
    res[2] = .4444444444444444*t28*t44+t1*t43+2.666666666666667*
        t26*t27;
    res[3] = t1*t49+1.333333333333333*t27*t32;
    res[4] = t1*t55;
    res[5] = t1*(-1.0*t10*(-84.488939072*t14*t16*POW(grada,
        8.0)/POW(rhoa,13.66666666666667)+45.62962962962963*t12*t14*
        t16*t57-.1916444444444444*t12*t57+2.477037037037037e-4*t5*
        t56+365.836336*t14*t35*t58*t16-332.5128888888888*t14*t5*t56*
        t16)+t24*t25*(-.2613333333333334*t8*grada/POW(rhoa,4.333333333333333)-
        0.1792*t35*t60*t58-.8885333333333333*t12*t21*t57-4.954074074074074e-4*
        t18*t3*t4*t5*t56+.5525333333333333*t5*t40*t56)+6.0*POW(t23,
        3.0)*t25*t59-6.0*t23*t25*t39*t42+3.0*t17*t24*t42-6.0*t17*t38*
        t39+3.0*t23*t24*t37)-.2962962962962963*t28/POW(rhoa,1.666666666666667)+
        1.333333333333333*t26*t44+4.0*t27*t43;
    res[6] = t1*(-1.0*t10*(63.366704304*t14*t16*POW(grada,
        7.0)/POW(rhoa,12.66666666666667)-19.55555555555556*t14*t16*
        t34*grada+.08213333333333332*grada*t34-1.3511111111111113e-4*
        t29*t33-245.495436*t14*t45*t36*t16+190.1466666666666*t14*t29*
        t33*t16)+6.0*t25*t31*t38*t59-4.0*t23*t25*t39*t48+2.0*t17*t24*
        t48+2.0*t23*t24*t47-2.0*t25*t31*t39*t42+t24*t25*(0.0784*t8*
        t41+2.702222222222222e-4*t18*t29*t3*t33*t4+0.1344*t45*t60*
        t36+0.4144*grada*t21*t34-0.3472*t29*t40*t33)-2.0*t30*t38*t39-
        4.0*t17*t23*t31*t39+t31*t24*t37+t42*t24*t30)+2.666666666666667*
        t27*t49+.4444444444444444*t32*t44;
    res[7] = t1*(-1.0*t10*(-47.525028228*t14*t16*t35/POW(rhoa,
        11.66666666666667)+158.849988*t14*t5*t46*t16+5.333333333333333*
        t13*t14*t16-96.536*t14*t12*t11*t16-0.0224*t13+6.399999999999999e-5*
        t11*t12)+6.0*t23*t25*t52*t59-2.0*t23*t25*t39*t54-2.0*t17*t39*
        t52+t23*t24*t51-4.0*t25*t31*t39*t48+2.0*t24*t30*t48+2.0*t24*
        t31*t47+t24*t25*(-0.1008*t5*t60*t46-1.2799999999999997e-4*
        t11*t12*t18*t3*t4-0.1344*t21*t13+0.2016*t12*t40*t11)-4.0*t23*
        t30*t31*t39+t54*t24*t17)+1.333333333333333*t27*t55;
    res[8] = t1*(-1.0*t10*(35.643771171*t14*t16*t45/POW(rhoa,
        10.66666666666667)-2.4e-5*t6*grada+39.492*t14*grada*t6*t16-
        97.47612899999999*t14*t29*t50*t16)+t24*t25*(1.2e-5*t18*t4*
        t53*t6*grada-0.1008*grada*t40*t6+0.0756*t29*t60*t50)+6.0*t25*
        POW(t31,3.0)*t59-6.0*t25*t31*t39*t54+3.0*t24*t30*t54-6.0*t30*
        t39*t52+3.0*t24*t31*t51);

}

static void
pw91x_third(FunThirdFuncDrv *ds, real factor, const FunDensProp* dp)
{
    real res[9];
 
    pw91x_third_helper(dp->rhoa, dp->grada, res);

    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];

    ds->df2000 += factor*res[2];
    ds->df1010 += factor*res[3];
    ds->df0020 += factor*res[4];

    ds->df3000 += factor*res[5];
    ds->df2010 += factor*res[6];
    ds->df1020 += factor*res[7];
    ds->df0030 += factor*res[8];


    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pw91x_third_helper(dp->rhob, dp->gradb, res);

    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];

    ds->df0200 += factor*res[2];
    ds->df0101 += factor*res[3];
    ds->df0002 += factor*res[4];

    ds->df0300 += factor*res[5];
    ds->df0201 += factor*res[6];
    ds->df0102 += factor*res[7];
    ds->df0003 += factor*res[8];

}

static void
pw91x_fourth_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15, t16, t17, t18;
    real t19, t20, t21, t22, t23, t24, t25, t26;
    real t27, t28, t29, t30, t31, t32, t33, t34;
    real t35, t36, t37, t38, t39, t40, t41, t42;
    real t43, t44, t45, t46, t47, t48, t49, t50;
    real t51, t52, t53, t54, t55, t56, t57, t58;
    real t59, t60, t61, t62, t63, t64, t65, t66;
    real t67, t68, t69, t70, t71, t72, t73, t74;
    real t75, t76, t77, t78, t79, t80, t81, t82;
    real t83, t84, t85, t86, t87, t88;

    t1 = POW(rhoa,1.333333333333333);
    t2 = POW(3.0,.3333333333333333);
    t3 = POW(4.0,.3333333333333333);
    t4 = POW(M_PI,.3333333333333333);
    t5 = POW(grada,4.0);
    t6 = 1/POW(rhoa,5.333333333333333);
    t7 = 1/t1;
    t8 = ASINH(grada*t7);
    t9 = 0.0252*grada*t8*t7+6.666666666666666e-7*t3*t4*t5*
        t6/t2+1.0;
    t10 = 1/t9;
    t11 = 1/POW(rhoa,6.333333333333333);
    t12 = POW(grada,2.0);
    t13 = 1/POW(rhoa,3.666666666666667);
    t14 = 0.0042-.1388888888888889/(POW(36.0,.6666666666666666)*
        POW(M_PI,1.666666666666667));
    t15 = 1/POW(rhoa,2.666666666666667);
    t16 = 1/POW(2.718281828459045,1.6455*t12*t15);
    t17 = 5.333333333333333e-6*t11*t5+2.666666666666667*t12*
        t13*t14*t16-4.388*t14*t5*t11*t16-0.0112*t12*t13;
    t18 = 1/POW(3.0,1.333333333333333);
    t19 = 1.9999999999999998e-6*t18;
    t20 = SQRT(t12*t15+1.0);
    t21 = 1/t20;
    t22 = 1/POW(rhoa,2.333333333333333);
    t23 = -5.333333333333333*t11*t19*t3*t4*t5-0.0336*grada*
        t8*t22-0.0336*t12*t21*t13;
    t24 = 1/POW(t9,2.0);
    t25 = -9.999999999999999e-7*t5*t6-1.0*t12*t14*t15*t16+
        0.0042*t12*t15;
    t26 = t23*t24*t25-1.0*t10*t17;
    t27 = POW(rhoa,.3333333333333333);
    t28 = -1.5*t2/(t3*t4)-1.0*t10*t25;
    t29 = POW(grada,3.0);
    t30 = -2.0*t14*t15*t16*grada-3.9999999999999997e-6*t29*
        t6+3.291*t14*t29*t6*t16+0.0084*grada*t15;
    t31 = 0.0252*t8*t7+4.0*t19*t29*t3*t4*t6+0.0252*grada*
        t21*t15;
    t32 = t31*t24*t25-1.0*t10*t30;
    t33 = 1/POW(rhoa,7.333333333333333);
    t34 = 1/POW(rhoa,4.666666666666667);
    t35 = POW(grada,6.0);
    t36 = 1/POW(rhoa,10.0);
    t37 = -3.377777777777778e-5*t33*t5-9.777777777777779*
        t12*t14*t16*t34+.04106666666666666*t12*t34-19.254544*t14*t35*
        t36*t16+39.492*t14*t5*t33*t16;
    t38 = POW(t23,2.0);
    t39 = 1/POW(t9,3.0);
    t40 = 1/POW(t20,3.0);
    t41 = 1/POW(rhoa,3.333333333333333);
    t42 = 6.755555555555556e-5*t18*t3*t33*t4*t5+0.0784*grada*
        t8*t41+0.168*t12*t21*t34-.04479999999999999*t5*t40*t33;
    t43 = -2.0*t25*t38*t39-1.0*t10*t37+t42*t24*t25+2.0*t17*
        t23*t24;
    t44 = 1/POW(rhoa,.6666666666666666);
    t45 = POW(grada,5.0);
    t46 = 1/POW(rhoa,9.0);
    t47 = 5.333333333333333*t13*t14*t16*grada+2.1333333333333332e-5*
        t11*t29+14.440908*t14*t45*t46*t16-26.328*t14*t29*t11*t16-0.0224*
        grada*t13;
    t48 = -4.2666666666666664e-5*t11*t18*t29*t3*t4-0.0336*
        t8*t22-0.1008*grada*t21*t13+0.0336*t29*t40*t11;
    t49 = -1.0*t10*t47-2.0*t23*t25*t31*t39+t23*t24*t30+t48*
        t24*t25+t31*t24*t17;
    t50 = 1/POW(rhoa,8.0);
    t51 = -1.2e-5*t12*t6+16.455*t14*t12*t6*t16-10.830681*
        t14*t5*t50*t16-2.0*t14*t15*t16+0.0084*t15;
    t52 = POW(t31,2.0);
    t53 = POW(4.0,1.333333333333333);
    t54 = 6.e-6*t12*t18*t4*t53*t6-0.0252*t12*t40*t6+0.0504*
        t21*t15;
    t55 = -2.0*t25*t39*t52-1.0*t10*t51+2.0*t24*t30*t31+t54*
        t24*t25;
    t56 = 1/POW(rhoa,8.333333333333334);
    t57 = 1/POW(rhoa,5.666666666666667);
    t58 = POW(grada,8.0);
    t59 = 1/POW(rhoa,13.66666666666667);
    t60 = 1/POW(rhoa,11.0);
    t61 = 45.62962962962963*t12*t14*t16*t57-.1916444444444444*
        t12*t57+2.477037037037037e-4*t5*t56+365.836336*t14*t35*t60*
        t16-84.488939072*t14*t58*t59*t16-332.5128888888888*t14*t5*
        t56*t16;
    t62 = POW(t23,3.0);
    t63 = 1/POW(t9,4.0);
    t64 = 1/POW(t20,5.0);
    t65 = 1/POW(rhoa,4.333333333333333);
    t66 = -.2613333333333334*grada*t8*t65-0.1792*t35*t64*
        t60-.8885333333333333*t12*t21*t57-4.954074074074074e-4*t18*
        t3*t4*t5*t56+.5525333333333333*t5*t40*t56;
    t67 = 6.0*t25*t62*t63-1.0*t10*t61-6.0*t23*t25*t39*t42+
        3.0*t17*t24*t42-6.0*t17*t38*t39+3.0*t23*t24*t37+t66*t24*t25;
    t68 = 1/
        POW(rhoa,1.666666666666667);
    t69 = POW(grada,7.0);
    t70 = 1/POW(rhoa,12.66666666666667);
    t71 = -19.55555555555556*t14*t16*t34*grada+.08213333333333332*
        grada*t34-1.3511111111111113e-4*t29*t33+63.366704304*t14*t69*
        t70*t16-245.495436*t14*t45*t36*t16+190.1466666666666*t14*t29*
        t33*t16;
    t72 = 0.0784*t8*t41+2.702222222222222e-4*t18*t29*t3*t33*
        t4+0.1344*t45*t64*t36+0.4144*grada*t21*t34-0.3472*t29*t40*
        t33;
    t73 = -1.0*t10*t71+6.0*t25*t31*t38*t63-4.0*t23*t25*t39*
        t48+2.0*t17*t24*t48+2.0*t23*t24*t47-2.0*t25*t31*t39*t42-2.0*
        t30*t38*t39-4.0*t17*t23*t31*t39+t31*t24*t37+t42*t24*t30+t72*
        t24*t25;
    t74 = 1/POW(rhoa,11.66666666666667);
    t75 = -47.525028228*t14*t35*t74*t16+158.849988*t14*t5*
        t46*t16+5.333333333333333*t13*t14*t16-96.536*t14*t12*t11*t16-
        0.0224*t13+6.399999999999999e-5*t11*t12;
    t76 = -0.1008*t5*t64*t46-1.2799999999999997e-4*t11*t12*
        t18*t3*t4-0.1344*t21*t13+0.2016*t12*t40*t11;
    t77 = -1.0*t10*t75+6.0*t23*t25*t52*t63-2.0*t23*t25*t39*
        t54-2.0*t17*t39*t52+t23*t24*t51-4.0*t25*t31*t39*t48+2.0*t24*
        t30*t48+2.0*t24*t31*t47-4.0*t23*t30*t31*t39+t76*t24*t25+t54*
        t24*t17;
    t78 = 1/POW(rhoa,10.66666666666667);
    t79 = -2.4e-5*t6*grada+35.643771171*t14*t45*t78*t16+39.492*
        t14*grada*t6*t16-97.47612899999999*t14*t29*t50*t16;
    t80 = POW(t31,3.0);
    t81 = 1.2e-5*t18*t4*t53*t6*grada-0.1008*grada*t40*t6+
        0.0756*t29*t64*t50;
    t82 = 6.0*t25*t63*t80-1.0*t10*t79-6.0*t25*t31*t39*t54+
        3.0*t24*t30*t54-6.0*t30*t39*t52+3.0*t24*t31*t51+t81*t24*t25;
    t83 = 1/
        POW(rhoa,9.333333333333334);
    t84 = 1/POW(rhoa,6.666666666666667);
    t85 = 1/POW(rhoa,14.66666666666667);
    t86 = 1/POW(rhoa,12.0);
    t87 = 1/POW(t9,5.0);
    t88 = 1/POW(t20,7.0);

   /* code */
    res[0] = 1.333333333333333*t27*t28+t1*t26;
    res[1] = t1*t32;
    res[2] = .4444444444444444*t28*t44+t1*t43+2.666666666666667*
        t26*t27;
    res[3] = t1*t49+1.333333333333333*t27*t32;
    res[4] = t1*t55;
    res[5] = -.2962962962962963*t28*t68+t1*t67+1.333333333333333*
        t26*t44+4.0*t27*t43;
    res[6] = t1*t73+2.666666666666667*t27*t49+.4444444444444444*
        t32*t44;
    res[7] = t1*t77+1.333333333333333*t27*t55;
    res[8] = t1*t82;
    res[9] = t1*(-1.0*t10*(-370.737464647936*t14*t16*POW(grada,
        10.0)/POW(rhoa,17.33333333333333)-258.5679012345679*t12*t14*
        t16*t84+1.085985185185185*t12*t84-.002064197530864197*t5*t83-
        5483.266252444444*t14*t35*t86*t16+2759.972009685333*t14*t58*
        t85*t16+2971.163555555555*t14*t5*t83*t16)-24.0*POW(t23,4.0)*
        t25*t87+t24*t25*(4.181333333333333*t35*t64*t86-1.194666666666667*
        t58*t88*t85+5.383466666666667*t12*t21*t84+.004128395061728395*
        t18*t3*t4*t5*t83-5.789155555555556*t5*t40*t83+1.132444444444445*
        grada*t8*t6)-8.0*t23*t25*t39*t66+4.0*t17*t24*t66+24.0*t17*
        t62*t63+36.0*t25*t38*t42*t63+4.0*t23*t24*t61-6.0*t25*t39*POW(t42,
        2.0)-24.0*t17*t23*t39*t42+6.0*t24*t37*t42-12.0*t37*t38*t39)-
        1.185185185185185*t26*t68+5.333333333333333*t27*t67+2.666666666666667*
        t43*t44+.4938271604938271*t15*t28;
    res[10] = t1*(-1.0*t10*(278.053098485952*t14*t16*POW(grada,
        9.0)/POW(rhoa,16.33333333333333)+91.25925925925925*t14*t16*
        t57*grada-.3832888888888888*grada*t57+9.908148148148148e-4*
        t29*t56+3289.317933333333*t14*t45*t60*t16-1879.878894352*t14*
        t69*t59*t16-1480.218666666666*t14*t29*t56*t16)-24.0*t25*t31*
        t62*t87-6.0*t23*t25*t39*t72+3.0*t17*t24*t72+3.0*t23*t24*t71-
        2.0*t25*t31*t39*t66+t24*t25*(-.2613333333333334*t8*t65-2.7328*
        t45*t64*t60+.8959999999999999*t69*t88*t59-2.0384*grada*t21*
        t57+3.098666666666666*t29*t40*t56-0.00198162962962963*t18*
        t29*t3*t4*t56)+6.0*t30*t62*t63+18.0*t25*t38*t48*t63+18.0*t23*
        t25*t31*t42*t63+18.0*t17*t31*t38*t63+t31*t24*t61-6.0*t25*t39*
        t42*t48-12.0*t17*t23*t39*t48+3.0*t24*t37*t48+3.0*t24*t42*t47-
        6.0*t38*t39*t47-6.0*t17*t31*t39*t42-6.0*t23*t30*t39*t42-6.0*
        t23*t31*t37*t39+t66*t24*t30)+4.0*t27*t73-.2962962962962963*
        t32*t68+1.333333333333333*t44*t49;
    res[11] = t1*(-1.0*t10*(-208.539823864464*t14*t16*t58/
        POW(rhoa,15.33333333333333)-19.55555555555556*t14*t16*t34+
        .08213333333333332*t34-4.053333333333333e-4*t12*t33+1251.492410004*
        t14*t35*t70*t16-1853.24986*t14*t5*t36*t16+634.7973333333333*
        t14*t12*t33*t16)-24.0*t25*t38*t52*t87-4.0*t23*t25*t39*t76+
        2.0*t17*t24*t76+2.0*t23*t24*t75-4.0*t25*t31*t39*t72+2.0*t24*
        t30*t72+2.0*t24*t31*t71+t24*t25*(-.6719999999999998*t35*t88*
        t70+8.106666666666666e-4*t12*t18*t3*t33*t4+1.7136*t5*t64*t36+
        0.4928*t21*t34-1.456*t12*t40*t33)+6.0*t25*t38*t54*t63+6.0*
        t25*t42*t52*t63+12.0*t17*t23*t52*t63+24.0*t23*t25*t31*t48*
        t63+12.0*t30*t31*t38*t63-2.0*t25*t39*t42*t54-4.0*t17*t23*t39*
        t54-2.0*t37*t39*t52-2.0*t38*t39*t51+t42*t24*t51-4.0*t25*t39*
        POW(t48,2.0)+4.0*t24*t47*t48-8.0*t17*t31*t39*t48-8.0*t23*t30*
        t39*t48-8.0*t23*t31*t39*t47-4.0*t30*t31*t39*t42+t54*t24*t37)+
        2.666666666666667*t27*t77+.4444444444444444*t44*t55;
    res[12] = t1*(-1.0*t10*(156.404867898348*t14*t16*t69/
        POW(rhoa,14.33333333333333)+1.2799999999999997e-4*t11*grada-
        807.9254798759999*t14*t45*t74*t16+953.099928*t14*t29*t46*t16-
        210.624*t14*grada*t11*t16)+t24*t25*(-2.56e-4*t11*t18*t3*t4*
        grada+0.504*t45*t88*t74-1.008*t29*t64*t46+0.5376*grada*t40*
        t11)-24.0*t23*t25*t80*t87-2.0*t23*t25*t39*t81+6.0*t17*t63*
        t80+t23*t24*t79-6.0*t25*t31*t39*t76+3.0*t24*t30*t76+3.0*t24*
        t31*t75+18.0*t23*t25*t31*t54*t63+18.0*t25*t48*t52*t63+18.0*
        t23*t30*t52*t63-6.0*t25*t39*t48*t54+3.0*t24*t47*t54-6.0*t17*
        t31*t39*t54-6.0*t23*t30*t39*t54-6.0*t39*t47*t52+3.0*t24*t48*
        t51-6.0*t23*t31*t39*t51-12.0*t30*t31*t39*t48+t81*t24*t17)+
        1.333333333333333*t27*t82;
    res[13] = t1*(-1.0*t10*(-117.303650923761*t14*t16*t35/
        POW(rhoa,13.33333333333333)-2.4e-5*t6+499.0127963939999*t14*
        t5*t78*t16+39.492*t14*t6*t16-422.3965589999999*t14*t12*t50*
        t16)-24.0*t25*POW(t31,4.0)*t87-8.0*t25*t31*t39*t81+4.0*t24*
        t30*t81+24.0*t30*t63*t80+4.0*t24*t31*t79+t24*t25*(-0.378*t5*
        t88*t78+1.2e-5*t18*t4*t53*t6-0.1008*t40*t6+0.5292*t12*t64*
        t50)+36.0*t25*t52*t54*t63-6.0*t25*t39*POW(t54,2.0)+6.0*t24*
        t51*t54-24.0*t30*t31*t39*t54-12.0*t39*t51*t52);

}

static void
pw91x_fourth(FunFourthFuncDrv *ds, real factor, const FunDensProp* dp)
{
    real res[14];
 
    pw91x_fourth_helper(dp->rhoa, dp->grada, res);

    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];

    ds->df2000 += factor*res[2];
    ds->df1010 += factor*res[3];
    ds->df0020 += factor*res[4];

    ds->df3000 += factor*res[5];
    ds->df2010 += factor*res[6];
    ds->df1020 += factor*res[7];
    ds->df0030 += factor*res[8];

    ds->df4000 += factor*res[9];
    ds->df3010 += factor*res[10];
    ds->df2020 += factor*res[11];
    ds->df1030 += factor*res[12];
    ds->df0040 += factor*res[13];


    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pw91x_fourth_helper(dp->rhob, dp->gradb, res);

    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];

    ds->df0200 += factor*res[2];
    ds->df0101 += factor*res[3];
    ds->df0002 += factor*res[4];

    ds->df0300 += factor*res[5];
    ds->df0201 += factor*res[6];
    ds->df0102 += factor*res[7];
    ds->df0003 += factor*res[8];

    ds->df0400 += factor*res[9];
    ds->df0301 += factor*res[10];
    ds->df0202 += factor*res[11];
    ds->df0103 += factor*res[12];
    ds->df0004 += factor*res[13];

}
