#ifndef CMDLINE_HIST_H
#define CMDLINE_HIST_H

#include <qstringlist.h>
#include <qmap.h>

/*!
 * \typedef QMap<QString,QStringList*> ChatHistories
 * Key is generated by CmdLine::chatUniqKey"("Chat* chat")" method.
 * Value is pointer to string list that represents list of input history entries.
 */
typedef QMap<QString,QStringList*> ChatHistories;
/*!
 * \typedef QMap<QString,uint*> ChatHistoryIndexes
 * Key is generated by CmdLine::chatUniqKey"("Chat* chat")" method.
 * Value is pointer to unsigned integer that is index of current history entry read for the chat window.
 */
typedef QMap<QString,uint*> ChatHistoryIndexes;

/*!
 * This class manages chats input history. It remembers values entered in input
 * and gives next/previous entries just by method call. Developer does not need
 * to worry about "which entry should be read now" - he just call getNext"("QString key")"
 * and getPrev"("QString key, QString msg")" methods.
 * \class CmdLineHistory
 * \brief Chat input history.
 */
class CmdLineHistory
{
	private:
		/*!
		 * \var ChatHistories chatHistories
		 * Keeps chats input history entries.
		 */
		ChatHistories chatHistories;
		/*!
		 * \var ChatHistoryIndexes chatHistoryIndexes
		 * Keeps chat input history indexes. They are needed
		 * for returning requested entries.
		 */
		ChatHistoryIndexes chatHistoryIndexes;
		/*!
		 * \var bool lastTemp
		 * Keeps information if last element in list is just templorary added
		 * to let user get back input value before using history.
		 * This last element should be erased after sending message.
		 */
		bool lastTemp;

		/*!
		 * \var uint histSize
		 * This member allows to control size of each input history.
		 * It describes number of history entries to remember.
		 */
		uint histSize;

		/*!
		 * \fn void addLine(QString key, QString msg)
		 * It appends new entry to history and automaticaly prevents history to grow
		 * bigger than <i>histSize</i> value.
		 * \param key Chat unique key generated with CmdLine::chatUniqKey"("Chat* chat")".
		 * \param msg Message to be added as a new history entry.
		 */
		void addLine(QString key, QString msg);

	public:
		/*!
		 * \fn CmdLineHistory()
		 * Default constructor. Sets default history size, which is 30.
		 */
		CmdLineHistory() : histSize(30) {}

		/*!
		 * \fn QString getNext(QString key)
		 * Returns next (newer) entry from the history for chat with given key.
		 * \param key Chat unique key generated with CmdLine::chatUniqKey"("Chat* chat")".
		 * \return Requested entry string or <code>QString::null</code> in case of no entries found in the history,
		 * 	or there is no more entries on the front of list.
		 */
		QString getNext(QString key);

		/*!
		 * \fn QString getPrev(QString key, QString msg)
		 * Returns previous (older) entry from the history for chat with given key.
		 * \param key Chat unique key generated with CmdLine::chatUniqKey"("Chat* chat")".
		 * \param msg This should be current input value. Method will remember this value
		 * 	to let user get back to this value. This value is marked as templorary and will be erased
		 * 	just after sending any other message (by calling messageSent"("QString key, QString msg")").
		 * \return Requested entry string or <code>QString::null</code> in case of no entries found in the history,
		 * or there is no more entries on the back of list.
		 */
		QString getPrev(QString key, QString msg);

		/*!
		 * \fn void messageSent(QString key, QString msg)
		 * This should be called when any message is sent in chat window. It adds new value to history list.
		 * \param key Chat unique key generated with CmdLine::chatUniqKey"("Chat* chat")".
		 * \param msg Value to add to the list.
		 */
		void messageSent(QString key, QString msg);

		/*!
		 * \fn void reset(QString key)
		 * If for some reason history index for given chat window should be reseted (should point to last element in the list),
		 * then call this method.
		 * \param key Chat unique key generated with CmdLine::chatUniqKey"("Chat* chat")".
		 */
		void reset(QString key);
};

#endif
