/*
     This file is part of GNUnet.
     (C) 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/fs.c
 * @brief main file-sharing code of gnunet-gtk
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "fs.h"
#include "download.h"
#include "search.h"
#include "upload.h"
#include "collection.h"
#include "namespace.h"
#include <GNUnet/gnunet_fsui_lib.h>

struct FSUI_Context * ctx;

static void saveEventProcessor(void * arg) {
  const FSUI_Event * event = arg;

  DEBUG_BEGIN();
  switch (event->type) {
  case FSUI_search_result:
    displaySearchResult(&event->data.SearchResult.fi,
			event->data.SearchResult.searchURI,
			NULL);
    break;
  case FSUI_search_error:
    LOG(LOG_ERROR,
	_("Error while searching: %s\n"),
	event->data.message);
    break;
  case FSUI_download_aborted:
  case FSUI_download_suspending:
    /* ignore for now */
    break;
  case FSUI_download_progress:
    displayDownloadUpdate(event->data.DownloadProgress.uri,
			  event->data.DownloadProgress.completed,
			  event->data.DownloadProgress.last_block,
			  event->data.DownloadProgress.last_size);
    break;
  case FSUI_download_complete:
    displayDownloadComplete(event->data.DownloadProgress.uri,
			    event->data.DownloadProgress.filename);
    addLogEntry(_("Download `%s' complete"), 
		event->data.DownloadProgress.filename);	
    gnunetgtk_notify(NOTIFY_NORMAL,
		     _("Download `%s' complete"), 
		     event->data.DownloadProgress.filename);	
    break;
  case FSUI_download_error:
    BREAK();
    LOG(LOG_ERROR,
	_("Error while downloading: %s\n"),
	event->data.message);
    break;
  case FSUI_upload_progress:
    displayUploadUpdate(event->data.UploadProgress.main_filename,
			event->data.UploadProgress.filename,
			event->data.UploadProgress.completed,
			event->data.UploadProgress.total);
    displayUploadUpdate(event->data.UploadProgress.main_filename,
			event->data.UploadProgress.main_filename,
			event->data.UploadProgress.main_completed,
			event->data.UploadProgress.main_total);
    break;
  case FSUI_upload_complete:
    displayUploadComplete(event->data.UploadComplete.main_filename,
			  event->data.UploadComplete.filename,
			  event->data.UploadComplete.uri);
    addLogEntry(_("Upload `%s' complete"), 
		event->data.UploadComplete.filename);
    gnunetgtk_notify(NOTIFY_NORMAL,
		     _("Upload `%s' complete"), 
		     event->data.UploadComplete.filename);
    break;
  case FSUI_upload_error:
    LOG(LOG_ERROR,
	_("Error while uploading: %s\n"),
	event->data.message);
    gnunetgtk_notify(NOTIFY_NORMAL,
		     _("Error while uploading `%s'"), 
		     event->data.message);
    break;
  case FSUI_gnunetd_connected:
    LOG(LOG_MESSAGE,
	_("Connected to gnunetd.\n"));
    break;
  case FSUI_gnunetd_disconnected:
    LOG(LOG_MESSAGE,
	_("Disconnected from gnunetd.\n"));
    addLogEntry(_("Disconnected from gnunetd.\n"));
    break;
  default:
    BREAK();
    LOG(LOG_ERROR,
	_("Unhandled (unknown) FSUI event: %u.\n"),
	event->type);
    break;
  }
  DEBUG_END();
}

/**
 * FSUI event handler.
 */
static void eventProcessor(void * cls,
			   const FSUI_Event * event) {
  gtkSaveCall(&saveEventProcessor,
	      (void*) event);
}

void init_fs() {
  GtkWidget * tab;
  GtkWidget * book;
  gint num;

  tab
    = glade_xml_get_widget(getMainXML(),
			   "fsnotebook");
  gtk_widget_show(tab);
  book
    = glade_xml_get_widget(getMainXML(), "mainnotebook");
  num = gtk_notebook_get_current_page(GTK_NOTEBOOK(book));
  gtk_notebook_set_current_page(GTK_NOTEBOOK(book), 1);
  gtk_notebook_set_current_page(GTK_NOTEBOOK(book), num);

  ctx = FSUI_start("gnunet-gtk",
		   YES,
		   &eventProcessor,
		   NULL);
  fs_collection_start();
  fs_search_start();
  fs_download_start();
  fs_upload_start();
  fs_namespace_start();
}

void done_fs() {
  DEBUG_BEGIN();
  fs_download_stop();
  fs_search_stop();
  fs_collection_stop();
  fs_namespace_stop();
  fs_upload_stop();
  FSUI_stop(ctx);
  DEBUG_END();
}

/* end of fs.c */
