(in-package :contextl)

(defclass special-object (standard-object)
  ())

(defclass special-class (standard-class)
  (old-slot-definitions))

(defmethod validate-superclass
           ((class special-class)
            (superclass standard-class))
  t)

(defmethod initialize-instance :around
  ((class special-class) &rest initargs
   &key direct-superclasses)
  (declare (dynamic-extent initargs))
  (if (loop for superclass in direct-superclasses
            thereis (subtypep superclass 'special-object))
      (call-next-method)
    (apply #'call-next-method class
           :direct-superclasses
           (append direct-superclasses
                   (list (find-class 'special-object)))
           initargs)))

(defmethod reinitialize-instance :around
  ((class special-class) &rest initargs
   &key (direct-superclasses () direct-superclasses-p))
  (declare (dynamic-extent initargs))
  (if direct-superclasses-p
      (if (loop for superclass in direct-superclasses
                thereis (subtypep superclass 'special-object))
          (call-next-method)
        (apply #'call-next-method class
               :direct-superclasses
               (append direct-superclasses
                       (list (find-class 'special-object)))
               initargs))
    (call-next-method)))

(defgeneric slot-definition-specialp (slot)
  (:method ((slot slot-definition)) nil))

(defclass special-direct-slot-definition (standard-direct-slot-definition)
  ((specialp :initarg :special
             :initform nil
             :reader slot-definition-specialp)))

(defclass special-effective-slot-definition (standard-effective-slot-definition)
  ())

(defmethod slot-definition-specialp ((slot special-effective-slot-definition))
  t)

(defmethod direct-slot-definition-class
           ((class special-class) &key &allow-other-keys)
  (find-class 'special-direct-slot-definition))

(defvar *special-effective-slot-definition-class*)

(defmethod effective-slot-definition-class
           ((class special-class) &key &allow-other-keys)
  *special-effective-slot-definition-class*)

(defmethod compute-effective-slot-definition
           ((class special-class) name direct-slot-definitions)
  (declare (ignore name))
  (let ((*special-effective-slot-definition-class*
         (if (some #'slot-definition-specialp direct-slot-definitions)
             (find-class 'special-effective-slot-definition)
           (find-class 'standard-effective-slot-definition))))
    (call-next-method)))

(defmethod shared-initialize :after
  ((object special-object) slot-names &rest all-keys)
  "ensure that all initialized special slots are indeed bound to
   a special slot; circumvents possible optimizations in the
   initialization of standard-class objects"
  (declare (ignore slot-names all-keys))
  (loop with class = (class-of object)
        for slot in (class-slots class)
        for slot-name = (slot-definition-name slot)
        when (typep slot 'special-effective-slot-definition)
        do (with-symbol-access
             (when (slot-boundp object slot-name)
               (let ((slot-value (slot-value object slot-name)))
                 (unless (special-symbol-p slot-value)
                   (slot-makunbound object slot-name)
                   (without-symbol-access
                     (setf (slot-value object slot-name) slot-value))))))))

(defvar *slot-unbound-p* nil)

(defmethod slot-unbound :around ((class special-class) object slot-name)
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (if *slot-unbound-p*
      (call-next-method)
    (let ((slot (find slot-name (the list (class-slots class))
                      :test #'eq
                      :key #'slot-definition-name)))
      (if (typep slot 'special-effective-slot-definition)
          (with-symbol-access
            (setf (slot-value-using-class class object slot)
                  (make-special-symbol)))
        (call-next-method)))))

(defmethod slot-value-using-class
           ((class special-class) object (slot special-effective-slot-definition))
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (let ((slot-symbol (call-next-method)))
    (declare (type symbol slot-symbol))
    (cond (*symbol-access* slot-symbol)
          ((boundp slot-symbol) (symbol-value slot-symbol))
          (t (let ((*slot-unbound-p* t))
               (slot-unbound class object (slot-definition-name slot)))))))

(defmethod (setf slot-value-using-class)
           (new-value (class special-class) object (slot special-effective-slot-definition))
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (if *symbol-access*
      (call-next-method)
    (let ((slot-symbol (with-symbol-access (slot-value-using-class class object slot))))
      (setf (symbol-value (the symbol slot-symbol)) new-value))))

(defmethod slot-boundp-using-class
           ((class special-class) object (slot special-effective-slot-definition))
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (if *symbol-access*
      (call-next-method)
    (let ((slot-symbol (with-symbol-access (slot-value-using-class class object slot))))
      (boundp (the symbol slot-symbol)))))

(defmethod slot-makunbound-using-class
           ((class special-class) object (slot special-effective-slot-definition))
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (if *symbol-access*
      (call-next-method)
    (let ((slot-symbol (with-symbol-access (slot-value-using-class class object slot))))
      (makunbound (the symbol slot-symbol))
      object)))

(defmethod reinitialize-instance :before
  ((class special-class) &key)
  (when (class-finalized-p class)
    (setf (slot-value class 'old-slot-definitions)
          (class-slots class))))

(defmethod finalize-inheritance :after
  ((class special-class))
  "ensure that special slots remain special after class redefinition
   (there is no protocol for collapsing multiple values in different
   dynamic scopes for the same special slot); make instances obsolete
   when non-special slots have been turned into special slots"
  (when (slot-boundp class 'old-slot-definitions)
    (assert (loop for old-slot in (slot-value class 'old-slot-definitions)
                  for new-slot = (find (slot-definition-name old-slot)
                                       (class-slots class)
                                       :test #'eq
                                       :key #'slot-definition-name)
                  always
                  #+(and allegro (not (version>= 7 0)))
                  (cond ((null new-slot) t)
                        (t (eql (typep old-slot 'special-effective-slot-definition)
                                (typep new-slot 'special-effective-slot-definition))))
                  #-(and allegro (not (version>= 7 0)))
                  (cond ((null new-slot) t)
                        ((typep old-slot 'special-effective-slot-definition)
                         (typep new-slot 'special-effective-slot-definition))
                        (t (when (typep new-slot 'special-effective-slot-definition)
                             (make-instances-obsolete class))
                           t)))
        ()
      #+(and allegro (not (version>= 7 0))) "The (non-)special slots in class ~S must remain (non-)special."
      #-(and allegro (not (version>= 7 0))) "The special slots in class ~S must remain special."
      (class-name class))
    (slot-makunbound class 'old-slot-definitions)))
