(in-package :contextl)

(defvar *layered-function-definers*
  (make-enclosing-package "LAYERED-FUNCTION-DEFINERS"))

(defun function-name-p (name)
  (or (symbolp name)
      (and (consp name)
           (eq (car name) 'setf)
           (symbolp (cadr name))
           (null (cddr name)))))

(defun get-layered-function-definer-name (name)
  (cond ((symbolp name)
         (enclose-symbol name *layered-function-definers*))
        ((consp name)
         (unless (function-name-p name)
           (error "Illegal function name: ~S." name))
         `(setf ,(enclose-symbol (cadr name) *layered-function-definers*)))
        (t (error "Illegal function name: ~S." name))))

(defun parse-method-body (form body)
  (let* ((in-layerp (eq (car body) :in-layer))
         (layer-spec (if in-layerp (cadr body) 't)))
    (when (consp layer-spec)
      (unless (null (cddr layer-spec))
        (error "Incorrect :in-layer specification in ~S." form)))
    (loop with layer = (if (atom layer-spec)
                         layer-spec
                         (cadr layer-spec))
          with layer-arg = (if (atom layer-spec)
                             (gensym "LAYER-ARG-")
                             (car layer-spec))
          for tail on (if in-layerp (cddr body) body)
          until (listp (car tail))
          collect (car tail) into qualifiers
          finally
          (when (member :in-layer qualifiers)
            (error "Incorrect occurrence of :in-layer in ~S. Must occur before qualifiers." form))
          (return (values layer-arg layer qualifiers (car tail) (cdr tail))))))

(defun prepare-layer (layer)
  (if (symbolp layer)
    (defining-layer layer)
    layer))

(defmacro define-layered-function (name (&rest args) &body options)
  (let ((definer (get-layered-function-definer-name name)))
    (with-unique-names (layer-arg rest-arg)
      `(progn
         (defgeneric ,definer (,layer-arg ,@args)
           (:argument-precedence-order 
            ,@(let ((argument-precedence-order (assoc :argument-precedence-order options)))
                (if argument-precedence-order
                  (cdr argument-precedence-order)
                  (required-args args)))
            ,layer-arg)
           ,@(loop for option in (remove :argument-precedence-order options :key #'car)
                   if (eq (car option) :method)
                   collect (multiple-value-bind
                               (layer-arg layer qualifiers args method-body)
                               (parse-method-body option (cdr option))
                             `(:method ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
                               ,@method-body))
                   else collect option))
         (declaim (inline ,name))
         (defun ,name (&rest ,rest-arg)
           (declare #-openmcl (dynamic-extent ,rest-arg)
                    (optimize (speed 3) (debug 0) (safety 0)
                              (compilation-speed 0)))
           (apply #',definer (layer-context-prototype *active-context*) ,rest-arg))
         #',definer))))

(defmacro define-layered-method (&whole form name &body body)
  (multiple-value-bind
      (layer-arg layer qualifiers args method-body)
      (parse-method-body form body)
    `(defmethod ,(get-layered-function-definer-name name)
                ,@qualifiers ((,layer-arg ,(prepare-layer layer)) ,@args)
       ,@method-body)))

(defun ensure-layered-function
       (name
        &rest initargs
        &key (lambda-list () lambda-list-p)
        (argument-precedence-order (required-args lambda-list))
        &allow-other-keys)
  (declare (dynamic-extent initargs))
  (unless lambda-list-p
    (error "The layered function ~S must be initialized with a lambda list." name))
  (let ((gf (let ((layer-arg (gensym "LAYER-ARG-")))
              (apply #'ensure-generic-function
                     (get-layered-function-definer-name name)
                     :argument-precedence-order
                     `(,@argument-precedence-order ,layer-arg)
                     :lambda-list
                     `(,layer-arg ,@lambda-list)
                     initargs)))
        (rest-arg (gensym)))
    (setf (fdefinition name)
          (compile nil `(lambda (&rest ,rest-arg)
                          (declare (dynamic-extent ,rest-arg)
                                   (optimize (speed 3) (debug 0) (safety 0)
                                             (compilation-speed 0)))
                          (apply (the function ,gf)
                                 (layer-context-prototype *active-context*)
                                 ,rest-arg))))
    gf))

(defun ensure-layered-method
       (layered-function-name
        lambda-expression 
        &key
        #-(or allegro clisp cmu mcl sbcl) 
        (method-class
         (generic-function-method-class
          (fdefinition (get-layered-function-definer-name layered-function-name))))
        (in-layer (find-layer 't))
        (qualifiers ())
        (lambda-list (cadr lambda-expression))
        (specializers (required-args lambda-list (constantly (find-class 't)))))
  (let ((layer-arg (gensym "LAYER-ARG-")))
    (destructuring-bind
        (lambda (&rest args) &body body)
        lambda-expression
      (unless (eq lambda 'lambda)
        (error "Incorrect lambda expression: ~S." lambda-expression))
      (ensure-method (fdefinition (get-layered-function-definer-name layered-function-name))
                     `(lambda (,layer-arg ,@args) ,@body)
                     #-(or allegro clisp cmu mcl sbcl) :method-class
                     #-(or allegro clisp cmu mcl sbcl) method-class
                     :qualifiers qualifiers
                     :lambda-list `(,layer-arg ,@lambda-list)
                     :specializers (cons (prepare-layer in-layer) specializers)))))
