package GCPanel;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################
use utf8;
use Gtk2;
use GCBorrowings;
use GCGraphicComponents;

use strict;

{
    package GCMoviePanel;
    use base "Gtk2::Frame";
    
    sub new
    {
        my ($proto, $parent, $options) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new();

        bless ($self, $class);

        $self->{disableBorrowerChange} = 0;
        $self->{autoUpdate} = 1;
        $self->{previouslyHidden} = 0;

        $self->{parent} = $parent;
        $self->{lang} = $parent->{lang};
        $self->{options} = $options;
        $self->set_shadow_type('none');
        
        $self->{infoBox} = new Gtk2::VBox(1,0);
        $self->{viewAllBox} = new Gtk2::EventBox;
        $self->{viewAllVBox} = new Gtk2::VBox(0,0);
        $self->{viewAll} = new Gtk2::HBox(0,0);
        $self->{viewAllButton} = new Gtk2::Button->new_from_stock('gtk-refresh');
        $self->{viewAll}->pack_start($self->{viewAllButton},1,0,50);
        $self->{viewAllVBox}->pack_start($self->{viewAll},0,0,0);
        $self->{viewAllBox}->add($self->{viewAllVBox});
        $self->{viewAllButton}->signal_connect('clicked' => sub {
            $self->{parent}->allMovies;
        });

        return $self;
    }
    
    sub initShow
    {
        my ($self, $show, $hasToShow) = @_;
       
        $self->{show} = $show;
       
        if ($hasToShow)
        {
            $self->show_all;
            $self->{infoBox}->hide;
            return 1;
        }
        else
        {
            $self->{previouslyHidden} = 1;
            return 0;
        }
    }
    
    sub show
    {
        my $self = shift;
        
        $self->{mainBox}->show;
        $self->{infoBox}->hide;
        $self->setShowOption($self->{show}, 1) if $self->{previouslyHidden};
        $self->{previouslyHidden} = 0;
    }
    
    sub hide
    {
        my ($self, $filtering) = @_;
        
        $self->{infoBox}->show_all;
        if ($filtering)
        {
            $self->{warning}->set_markup('<b>'.$self->{lang}->{AllMoviesFiltered}.'</b>');
        }
        else
        {
            $self->{warning}->set_markup($self->{lang}->{Warning}) if !$filtering;
            $self->{viewAllBox}->hide;
        }
        $self->{mainBox}->hide;
    }
    
    sub showMe
    {
        my $self = shift;
        my $parent = $self->parent;
        while ($parent)
        {
           last if ($parent->isa('Gtk2::Window'));
           $parent = $parent->parent;
        }
        $parent->present if $parent;
    }
    
    sub getAsHash
    {
        my $self = shift;
        my $hash = {};
        $hash->{$_} = $self->$_ foreach(@{$self->{parent}->{movies}->{fields}});
        return $hash;
    }
    
    sub disableBorrowerChange
    {
        my $self = shift;
        $self->{disableBorrowerChange} = 1;
    }
    
    sub disableAutoUpdate
    {
        my $self = shift;
        $self->{autoUpdate} = 0;
    }
}

{
    package GCReadOnlyPanel;
    use base 'GCMoviePanel';

    use File::Basename;
    use Text::Wrap;

    sub new
    {
        my ($proto, $parent, $options) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent, $options);

        bless ($self, $class);

        my $bgColor;
        my $boxColor;
        my $fieldColor;
        my $titleColor;
        my $labelColor;
        my $titleTextColor;
        my $labelTextColor;
        my $fieldTextColor;
        my $activeTextColor;
        my $inactiveTextColor;
        
        $options->order(3) if (! -d $ENV{GCF_SHARE_DIR}.'/panel/');
        
        if ($options->order == 3)
        {
            #Classic layout
            $bgColor = new Gtk2::Gdk::Color(65535,65535,65535);
            $boxColor = new Gtk2::Gdk::Color(60000,60000,60000);
            $fieldColor = new Gtk2::Gdk::Color(45000,45000,45000);
            $titleColor = new Gtk2::Gdk::Color(65535,65535,65535);
            $labelColor = new Gtk2::Gdk::Color(0,0,0);
            $titleTextColor = '#000000';
            $labelTextColor = '#000000';
            $fieldTextColor = '#ffffff';
            $activeTextColor = '#000000';
            $inactiveTextColor = '#444444';
        }
        else
        {
            #Web site layout
            $bgColor = new Gtk2::Gdk::Color(65535,65535,65535);
            $boxColor = new Gtk2::Gdk::Color(61166,61166,61166);
            $fieldColor = new Gtk2::Gdk::Color(30583,34952,39321);
            $titleColor = new Gtk2::Gdk::Color(65535,55255,0);
            $labelColor = new Gtk2::Gdk::Color(7196,34438,61166);
            $titleTextColor = '#ffffff';
            $labelTextColor = '#1c86ee';
            $fieldTextColor = '#ffffff';
            $activeTextColor = '#000000';
            $inactiveTextColor = '#444444';            
        }

        #Item is made of:
        # [Automatic creation?, X, Y, Background color, span style, Table Name, Height, Pre Process?]
        # If (Pre Process == 1), there must be a method prepare_<fieldName>
        $self->{items} = {
            'title' => [0,1,0,$titleColor,"size='xx-large' weight='bold' foreground='".$titleTextColor."'",'header', 0, 0],
            'orig' =>  [0,1,1,$titleColor,"size='x-large' style='italic' foreground='".$titleTextColor."'",'header', 0, 0],

            'dateLabel' => [1,0,0,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'date' => [1,1,0,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 0],
            'directorLabel' => [1,0,1,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'director' => [1,1,1,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 0],
            'timeLabel' => [1,0,2,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'time' => [1,1,2,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 0],
            'natLabel' => [1,0,3,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'nat' => [1,1,3,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 0],
            'ageLabel' => [1,0,4,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'age' => [1,1,4,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 1],
            'typeLabel' => [1,0,5,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'info', 0, 0],
            'type' => [1,1,5,$fieldColor,"foreground='".$fieldTextColor."'",'info', 0, 1],

            'rating' => [0,0,0,0,"foreground='".$activeTextColor."'",'', 0, 1],
            'seenYes' => [0,0,0,0,"foreground='".$inactiveTextColor."'",'', 0, 0],
            'seenNo' => [0,0,0,0,"weight='bold' foreground='".$activeTextColor."'",'', 0, 0],
            'lendedYes' => [0,0,0,0,"foreground='".$inactiveTextColor."'",'', 0, 0],
            'lendedNo' => [0,0,0,0,"weight='bold' foreground='".$activeTextColor."'",'', 0, 0],
            
            'formatLabel' => [1,0,0,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'format' => [1,1,0,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 1],
            'placeLabel' => [1,0,1,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'place' => [1,1,1,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 0],
            'collectionLabel' => [1,0,2,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'collection' => [1,1,2,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 1],
            'audioLabel' => [1,0,3,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'audio' => [1,1,3,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 1],
            'subtLabel' => [1,0,4,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'subt' => [1,1,4,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 1],
            'videoLabel' => [1,0,5,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'perso', 0, 0],
            'video' => [1,1,5,$fieldColor,"foreground='".$fieldTextColor."'",'perso', 0, 0],

            'actorsLabel' => [1,0,0,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'details', 0, 0],
            'actors' => [1,1,0,$fieldColor,"foreground='".$fieldTextColor."'",'details', 45, 0],
            'synopsisLabel' => [1,0,1,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'details', 0, 0],
            'synopsis' => [1,1,1,$fieldColor,"foreground='".$fieldTextColor."'",'details', 80, 1],
            'commentLabel' => [1,0,2,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'details', 0, 0],
            'comment' => [1,1,2,$fieldColor,"foreground='".$fieldTextColor."'",'details', 80, 1],

            'borrowerLabel' => [1,0,0,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'lending', 0, 0],
            'borrower' => [1,1,0,$fieldColor,"foreground='".$fieldTextColor."'",'lending', 0, 0],
            'lendDateLabel' => [1,0,1,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'lending', 0, 0],
            'lendDate' => [1,1,1,$fieldColor,"foreground='".$fieldTextColor."'",'lending', 0, 0],
            'historyLabel' => [1,0,2,$boxColor,"weight='bold' foreground='".$labelTextColor."'",'lending', 0, 0],
            'historyHeader' => [0,0,0,$fieldColor,"weight='bold' foreground='".$fieldTextColor."'",'lending', 0, 0],
            'history' => [0,1,2,$fieldColor,"foreground='".$fieldTextColor."'",'lending', 0, 0],
        };

        $self->{tooltips} = $parent->{tooltips};

        $self->{vboxMain} = new Gtk2::VBox(0, 0);

        $self->{rating} = new Gtk2::Label;
        $self->{seen} = new Gtk2::Label;
        $self->{lended} = new Gtk2::Label;
        $self->{image} = new GCMovieImage($options, $parent->{defaultImage},1);

        $self->{tables} = {};
        $self->{tables}->{header} = new Gtk2::Table(4,3);
        $self->{tables}->{header}->set_col_spacings(0);
        $self->{tables}->{header}->set_row_spacings(0);
        $self->{tables}->{info} = new Gtk2::Table(7,2);
        $self->{tables}->{info}->set_col_spacings(2);
        $self->{tables}->{info}->set_row_spacings(2);
        $self->{tables}->{perso} = new Gtk2::Table(7,2);
        $self->{tables}->{perso}->set_col_spacings(2);
        $self->{tables}->{perso}->set_row_spacings(2);
        $self->{tables}->{details} = new Gtk2::Table(4,2);
        $self->{tables}->{details}->set_col_spacings(2);
        $self->{tables}->{details}->set_row_spacings(5);
        $self->{tables}->{lending} = new Gtk2::Table(3,2);
        $self->{tables}->{lending}->set_col_spacings(2);
        $self->{tables}->{lending}->set_row_spacings(2);
        $self->{tables}->{history} = new Gtk2::Table(0,3);
        $self->{tables}->{history}->set_col_spacings(2);
        $self->{tables}->{history}->set_row_spacings(2);
        foreach (keys %{$self->{tables}})
        {
            my $fillLabel = new Gtk2::Label;
            $fillLabel->set_name('fillLabel'.$_);
            $fillLabel->set_markup("<span size='x-small'> </span>");
            $self->{tables}->{$_}->attach($fillLabel, 1,2,0,1,'expand','fill',0,0);
            $fillLabel->hide;
        }

        for my $name (keys %{$self->{items}})
        {
            my $item = $self->{items}->{$name};
            next if ! $item->[0];

            $self->{$name} = new GCColourLabel($item->[3]);
            if ($name =~ /Label$/)
            {
                (my $langName = $name) =~ s/Label$//;
                my $label = $parent->{lang}->{Displayed}->{$langName};
                $self->$name($label);
            }
            $self->{$name}->set_selectable(0);
            $self->{$name}->set_alignment(0,0.5);
            $self->{$name}->set_padding(5,5);
            $self->{$name}->set_justify('fill');

            my $widget = $self->{$name};
            if ($item->[6])
            {
                $self->{'scroll'.$name} = new Gtk2::ScrolledWindow;
                $self->{'scroll'.$name}->set_border_width(0);
                $self->{'scroll'.$name}->set_shadow_type('none');
                $self->{'scroll'.$name}->set_policy('automatic', 'automatic');
                $self->{'scroll'.$name}->set_size_request(-1,$item->[6]);
                $self->{'scroll'.$name}->add_with_viewport($self->{$name});
                $self->{'scroll'.$name}->child->set_shadow_type('none');
                $widget = $self->{'scroll'.$name};
            }
            
            $self->{tables}->{$item->[5]}->attach($widget,
                                                  $item->[1], 
                                                  $item->[1] + 1,
                                                  $item->[2] + 1,
                                                  $item->[2] + 2,
                                                  'fill', 'fill', 0, 0);
        }
        
        my $item = $self->{items}->{history};
        $self->{tables}->{lending}->attach($self->{tables}->{history},
                                           $item->[1], 
                                           $item->[1] + 1,
                                           $item->[2] + 1,
                                           $item->[2] + 2,
                                           'expand', 'fill', 0, 0);

        $self->{title} = new GCColourLabel($self->{items}->{title}->[3]);
        $self->{orig} = new GCColourLabel($self->{items}->{orig}->[3]);
        my $floatingBox = new Gtk2::VBox(0,0);
        $floatingBox->pack_start($self->{rating},1,1,2);
        $floatingBox->pack_start($self->{seen},1,1,2);
        $floatingBox->pack_start($self->{lended},1,1,2);
        my $greyBox = new Gtk2::EventBox;
        $greyBox->modify_bg('normal', $boxColor);
        $greyBox->set_size_request(100,-1);
        $greyBox->add($floatingBox);
        my $titlesBox = new Gtk2::VBox(0,0);
        $titlesBox->pack_start($self->{title},1,1,0);
        $titlesBox->pack_start($self->{orig},1,1,0);
        my $titleColorBox = new Gtk2::EventBox;
        $titleColorBox->modify_bg('normal', $titleColor);
        $titleColorBox->set_size_request(-1,50);
        my $spacingBox = new Gtk2::VBox(0,0);
        $spacingBox->pack_start($titleColorBox,1,1,0);
        
        if ($options->order == 4)
        {
            #Web site layout
            $self->{title}->set_alignment(0,0.5);
            $self->{title}->set_padding(10,0);
            $self->{orig}->set_alignment(0,0.5);
            $self->{orig}->set_padding(10,0);
            my $titleImg = Gtk2::Image->new_from_file($ENV{GCF_SHARE_DIR}.'/panel/title_back.png');
            $self->{tables}->{header}->attach($titleImg,
                                              0, 1, 1, 3, 'fill', 'fill', 0, 0);
            my $bgColorBox1 = new Gtk2::EventBox;
            $bgColorBox1->modify_bg('normal', $bgColor);
            $bgColorBox1->set_size_request(-1,11);
            my $bgColorBox2 = new Gtk2::EventBox;
            $bgColorBox2->modify_bg('normal', $bgColor);
            $bgColorBox2->set_size_request(-1,8);
            $titlesBox->pack_start($bgColorBox1,0,0,0);
            $titlesBox->reorder_child($bgColorBox1,0);
            $titlesBox->pack_start($bgColorBox2,0,0,0);
            my $rightCorner = Gtk2::Image->new_from_file($ENV{GCF_SHARE_DIR}.'/panel/title_right.png');
            $spacingBox->pack_start($rightCorner,0,0,0);
            $spacingBox->reorder_child($rightCorner,0);
            $titleColorBox->set_size_request(-1,40);
            my $topImg = Gtk2::Image->new_from_file($ENV{GCF_SHARE_DIR}.'/panel/floating_top.png');
            my $bottomImg = Gtk2::Image->new_from_file($ENV{GCF_SHARE_DIR}.'/panel/floating_bottom.png');
            $floatingBox->pack_start($topImg,0,0,0);
            $floatingBox->reorder_child($topImg,0);
            $floatingBox->pack_start($bottomImg,0,0,0);
        }
        else
        {
            #Classic layout
            my $leftSpacer = new Gtk2::HBox;
            $leftSpacer->set_size_request(100,-1);
            $self->{tables}->{header}->attach($leftSpacer,
                                              0, 1, 1, 3, 'fill', 'fill', 0, 0);
        }
 
        $self->{tables}->{header}->attach($titlesBox,
                                          1, 2, 1, 3, 'fill', 'fill', 0, 0);
        $self->{tables}->{header}->attach($spacingBox,
                                          2, 3, 1, 2, 'fill', 'fill', 0, 0);
        $self->{tables}->{header}->attach($greyBox,
                                          2, 3, 2, 4, 'fill', 'fill', 0, 0);
 
        $self->{vboxMain}->pack_start($self->{tables}->{header}, 0, 0, 0);
        
        my $hboxPicInfos = new Gtk2::HBox(0,0);
        $hboxPicInfos->set_border_width(0);
        $hboxPicInfos->pack_start($self->{image}, 0, 0, 20);
        $hboxPicInfos->pack_start($self->{tables}->{info}, 1, 1, 20);
        
        $self->{expanderPerso} = new Gtk2::Expander($parent->{lang}->{PanelPrivateInfo});
        $self->{expanderPerso}->modify_fg('normal', $labelColor);
        $self->{expanderPerso}->get_label_widget->modify_fg('normal', $labelColor);
        $self->{expanderPerso}->modify_fg('prelight', $labelColor);
        $self->{expanderPerso}->get_label_widget->modify_fg('prelight', $labelColor);
        $self->{expanderPerso}->modify_bg('normal', $bgColor);
        $self->{expanderPerso}->get_label_widget->modify_bg('normal', $bgColor);
        $self->{expanderPerso}->modify_bg('prelight', $boxColor);
        $self->{expanderPerso}->get_label_widget->modify_bg('prelight', $boxColor);
        my $hboxPersoMargins = new Gtk2::HBox(0,0);
        $hboxPersoMargins->pack_start($self->{tables}->{perso},1,1,40);
        $self->{expanderPerso}->add($hboxPersoMargins);

        $self->{expanderLending} = new Gtk2::Expander($parent->{lang}->{PanelLendInfo});
        $self->{expanderLending}->modify_fg('normal', $labelColor);
        $self->{expanderLending}->get_label_widget->modify_fg('normal', $labelColor);
        $self->{expanderLending}->modify_fg('prelight', $labelColor);
        $self->{expanderLending}->get_label_widget->modify_fg('prelight', $labelColor);
        $self->{expanderLending}->modify_bg('normal', $bgColor);
        $self->{expanderLending}->get_label_widget->modify_bg('normal', $bgColor);
        $self->{expanderLending}->modify_bg('prelight', $boxColor);
        $self->{expanderLending}->get_label_widget->modify_bg('prelight', $boxColor);
        my $hboxLendingMargins = new Gtk2::HBox(0,0);
        $hboxLendingMargins->pack_start($self->{tables}->{lending},1,1,40);
        $self->{expanderLending}->add($hboxLendingMargins);

        $self->{vboxMain}->pack_start($hboxPicInfos, 0, 0, 0);
        $self->{vboxMain}->pack_start($self->{tables}->{details}, 0, 0, 5);
        $self->{vboxMain}->pack_start($self->{expanderPerso}, 0, 0, 5);
        $self->{vboxMain}->pack_start($self->{expanderLending}, 0, 0, 5);

        my $hboxMargins = new Gtk2::HBox(0,0);
        $hboxMargins->pack_start($self->{vboxMain}, 1, 1, 50);
        $self->{mainBox} = new Gtk2::EventBox;
        $self->{mainBox}->modify_bg('normal', $bgColor);
        $self->{mainBox}->add($hboxMargins);

        my $realMain = new Gtk2::VBox(0,0);
        $realMain->pack_start($self->{mainBox},1,1,0);

        $self->add($realMain);
        $self->show_all;

        $self->{warning} = new GCColourLabel($bgColor);

        $self->{infoBox}->pack_start($self->{warning},1,1,0);
        $self->{viewAllBox}->modify_bg('normal', $bgColor);
        $self->{infoBox}->pack_start($self->{viewAllBox},1,1,0);
        $realMain->pack_start($self->{infoBox},1,1,0);
        
        return $self;
    }
    
    sub deactivate
	{
        my $self = shift;
	}
   
    sub setShowOption
    {
        my ($self, $show, $hasToShow) = @_;
       
        return if ! $self->initShow($show, $hasToShow);

        foreach (keys %{$show})
        {
            $self->{$_.'Label'}->hide if (! $show->{$_}) && ($self->{$_.'Label'});
            $self->{$_}->hide if (! $show->{$_}) && ($self->{$_});
            $self->{'scroll'.$_}->hide if (! $show->{$_}) && $self->{items}->{$_}->[6];
        }

        $self->{tables}->{history}->hide if (! $show->{history});

        my $showExpander = 0;
        my @children = $self->{tables}->{perso}->get_children;
        foreach (@children)
        {
            next if $_->get_name =~ /^fillLabel/;
            if ($_->visible)
            {
                $showExpander = 1;
                last;
            }
        }
        $self->{expanderPerso}->visible($showExpander);
        
        $showExpander = 0;
        @children = $self->{tables}->{lending}->get_children;
        foreach (@children)
        {
            next if $_->get_name =~ /^fillLabel/;
            if ($_->visible)
            {
                $showExpander = 1;
                last;
            }
        }
        $self->{expanderLending}->visible($showExpander);
    }

    sub selectTitle
    {
        my $self = shift;
    }

    sub setBorrowers
    {
        my ($self, $values) = @_;
    }

    sub setFormats
    {
        my ($self, $values) = @_;
    }

    sub setHistory
    {
        my ($self, $values) = @_;
    }

    sub trailer
    {
        my $self = shift;

        if (@_)
        {
            my $value = shift;
            $self->{parent}->enablePlay($value ? 1 : 0);
            $self->{trailerValue} = $value;
        }
        else
        {
            return $self->{trailerValue};
        }
    }

    sub url
    {
        my $self = shift;

        if (@_)
        {
            my $value = shift;
            $self->{urlValue} = $value;
        }
        else
        {
            return $self->{urlValue};
        }
    }

    sub image
    {
        my $self = shift;

        if (@_)
        {
            $self->{image}->setImage(shift);
        }
        else
        {
            return $self->{image}->getImage;
        }
    }

    sub seen
    {
        my $self = shift;
        if (scalar @_)
        {
            my $text = shift;
            $self->{seenValue} = $text;
            my $style;
            if ($text)
            {
                $text = $self->{lang}->{PanelSeenYes};
                $style = $self->{items}->{seenYes}->[4];
            }
            else
            {
                $text = $self->{lang}->{PanelSeenNo};
                $style = $self->{items}->{seenNo}->[4];
            }
            $self->{seen}->set_markup("<span ".
                                            $style.
                                            ">".$text.'</span>');
        }
        else
        {
            return $self->{seenValue};
        }
        
    }
    
    sub borrower
    {
        my $self = shift;
        our $AUTOLOAD = 'borrower';
        if (scalar @_)
        {
            my $text = shift;
            $self->{borrowerValue} = $text;
            my $style;
            my $lended;
            if ($text eq 'none' || !$text)
            {
                $text = $self->{lang}->{PanelNobody};
                $lended = $self->{lang}->{PanelLendedNo};
                $style = $self->{items}->{lendedNo}->[4];
            }
            else
            {
                $lended = $self->{lang}->{PanelLendedYes};
                $style = $self->{items}->{lendedYes}->[4];
            }
            $self->AUTOLOAD($text,1);
            $self->{lended}->set_markup("<span ".
                                            $style.
                                            ">".$lended.'</span>');
        }
        else
        {
            return $self->{borrowerValue};
        }
    }

    sub history
    {
        my $self = shift;
        if (scalar @_)
        {
            my $text = shift;
            $self->{historyValue} = $text;
            
            my @children = $self->{tables}->{history}->get_children;
        
            foreach (@children)
            {
                $self->{tables}->{history}->remove($_);
                $_->destroy;
            }
            my @lines = split m/,/, $text;
            my $number = scalar @lines;

            $self->{tables}->{history}->resize($number+1,3);
            my $label1 = new GCColourLabel($self->{items}->{history}->[3]);
            $label1->set_padding(5,5);
            $label1->set_markup("<span ".
                                    $self->{items}->{historyHeader}->[4].
                                ">".$self->{lang}->{Panel}->{borrower}.'</span>');
            $self->{tables}->{history}->attach($label1,0,1,1,2,'fill','fill',0,0);                
            my $label2 = new GCColourLabel($self->{items}->{history}->[3]);
            $label2->set_padding(5,5);
            $label2->set_markup("<span ".
                                    $self->{items}->{historyHeader}->[4].
                                ">".$self->{lang}->{Panel}->{lendDate}.'</span>');
            $self->{tables}->{history}->attach($label2,1,2,1,2,'fill','fill',0,0);                
            my $label3 = new GCColourLabel($self->{items}->{history}->[3]);
            $label3->set_padding(5,5);
            $label3->set_markup("<span ".
                                    $self->{items}->{historyHeader}->[4].
                                ">".$self->{lang}->{PanelReturnDate}.'</span>');
            $self->{tables}->{history}->attach($label3,2,3,1,2,'fill','fill',0,0);                

            my $row = 2;
            foreach (@lines)
            {
                my @values = split m/:/;
                my $col = 0;
                for my $field (@values)
                {
                    my $label = new GCColourLabel($self->{items}->{history}->[3]);
                    $label->set_padding(5,5);
                    $label->set_markup("<span ".
                                            $self->{items}->{history}->[4].
                                        ">".$field.'</span>');
                    $self->{tables}->{history}->attach($label,$col,$col+1,$row,$row+1,'fill','fill',0,0);
                    $col++;
                }
                $row++;
            }
            $self->{tables}->{history}->show_all if ($self->{show}->{history});
        }
        else
        {
            return $self->{historyValue};
        }
    }

    sub number
    {
        my $self = shift;
        if (scalar @_)
        {
            my $text = shift;
            $self->{numberValue} = $text;
            $text = $self->{formatValue}." (".$text.")";
            $self->{'format'}->set_markup("<span ".
                                            $self->{items}->{'format'}->[4].
                                            ">".$text.'</span>')
        }
        else
        {
            return $self->{numberValue};
        }
    }
    
    sub rank
    {
        my $self = shift;
        if (scalar @_)
        {
            my $text = shift;
            $text = 0 if !$text;
            $self->{rankValue} = $text;
            return '' if !$self->{collectionValue};
            $text = $self->{collectionValue}." (".$text.")";
            $self->{'collection'}->set_markup("<span ".
                                            $self->{items}->{'collection'}->[4].
                                            ">".$text.'</span>')
        }
        else
        {
            return $self->{rankValue};
        }
    }

    sub prepare_type
    {
        my ($self, $value) = @_;

        $value =~ s/,*$//;
        $value =~ s/,([^ ])/, $1/g;
        return $value;
    }

    sub prepare_age
    {
        my ($self, $value) = @_;
        
        return $value                               if $value >= 10;
        return ''                                   if !$value;
        return $self->{lang}->{PanelAgeUnknown}     if $value == 1;
        return $self->{lang}->{PanelAgeNone}        if ($value > 1) && ($value < 5);
        return $self->{lang}->{PanelAgeParent}      if ($value > 4) && ($value < 10);
        
    }

    sub prepare_synopsis
    {
        my ($self, $value) = @_;

        $value =~ s/<br>/\n/g;
        return $value;
    }
    
    sub prepare_comment
    {
        my ($self, $value) = @_;

        $value =~ s/<br>/\n/g;
        return $value;
    }
    
    sub prepare_format
    {
        my ($self, $value) = @_;

        $value = $value." (".$self->{numberValue}.")" if $value;
        return $value;
    }
    
    sub prepare_collection
    {
        my ($self, $value) = @_;

        $value = $value." (".$self->{rankValue}.")" if $value;
        return $value;
    }

    sub prepare_audio
    {
        my ($self, $value) = @_;

        $value =~ s/;.*?(,|$)/$1/g;
        $value =~ s/,*$//;
        $value =~ s/,([^ ])/, $1/g;
        return $value;
    }
    
    sub prepare_subt
    {
        my ($self, $value) = @_;

        $value =~ s/,*$//;
        $value =~ s/,([^ ])/, $1/g;
        return $value;
    }
    
    sub prepare_rating
    {
        my ($self, $value) = @_;

        $value .= '/10';
        return $value;
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        if (scalar @_)
        {
            my $text = shift;
            my $alreadySaved = shift;
            $self->{$name.'Value'} = $text if ! $alreadySaved;
            if ($self->{items}->{$name}->[7])
            {
                my $prepare = 'prepare_'.$name;
                $text = $self->$prepare($text);
            }
            $text = ' ' if !$text;

            if ($self->{items}->{$name}->[6])
            {
                $Text::Wrap::columns = 80;
                $text =  wrap('', '', $text);
            }
            
            $self->{$name}->set_markup("<span ".
                                            $self->{items}->{$name}->[4].
                                            ">".$text.'</span>')
                if $self->{$name};
        }
        else
        {
            return $self->{$name.'Value'};
        }
    }
}

{
    package GCFormPanel;

    use base 'GCMoviePanel';

    sub new
    {
        my ($proto, $parent, $options, $readonly, $show) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent, $options);

        bless ($self, $class);

        $self->{defaultImage} = $parent->{defaultImage};

        $self->{tooltips} = $parent->{tooltips};

        $self->{mainBox} = new Gtk2::VBox(0, 0);

        my $hboxTitle = new Gtk2::HBox(0,0);
        $self->{id} = Gtk2::Entry->new;
        $self->{id}->set_width_chars(5);
        $self->{title} = Gtk2::Entry->new; 
        $self->{title}->signal_connect('activate', \&searchMovie, $self) if ! $readonly;
        $self->{searchButton} = Gtk2::Button->new_from_stock('gtk-jump-to');
        $self->{searchButton}->signal_connect('clicked', \&searchMovie, $self);
        $self->{tooltips}->set_tip($self->{searchButton}, $parent->{lang}->{PanelSearchTip}, '');
        $hboxTitle->pack_start($self->{id}, 0, 0, 2);
        $hboxTitle->pack_start($self->{title}, 1, 1, 2);
        $hboxTitle->pack_start($self->{searchButton}, 0, 0, 0) if ! $readonly;

        my $framePerso = new Gtk2::Frame();
        $framePerso->set_border_width(2);
        $framePerso->set_label_align(1.0, 0.0);
        my $vboxPerso = new Gtk2::VBox(0,0);
        $vboxPerso->set_border_width(2);

        $self->{seen} = Gtk2::CheckButton->new($parent->{lang}->{Panel}->{seen});
        $self->{ratingLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{rating});
        my $adj1 = Gtk2::Adjustment->new(5, 0, 10, 1, 1, 0) ;
        $self->{rating} = Gtk2::SpinButton->new($adj1, 0, 0);
        $self->{rating}->set_numeric(1);

        $self->{formatLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{format});
		$self->{format} = new Gtk2::OptionMenu();
		$self->{formatMenu} = new Gtk2::Menu();
        $options->formats('DVD,DVDR,CD,S-VCD,VHS') if (! $options->exists('formats'));
        my @formats = split m/,/, $options->formats;
		my $format;
		foreach $format(@formats)
		{
		    my $item = Gtk2::MenuItem->new_with_mnemonic($format);
		    $self->{formatMenu}->append( $item) ;
		}
		$self->{formats} = \@formats;
		$self->{format}->set_menu($self->{formatMenu}) ;

        $self->{numberLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{number});
        my  $adj2 = Gtk2::Adjustment->new(1, 1, 9999, 1, 1, 0) ;
        $self->{number} = Gtk2::SpinButton->new($adj2, 0, 0);
        $self->{number}->set_numeric(1);
        $self->{placeLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{place});
        $self->{place} = Gtk2::Combo->new;
        ($self->{place}->get_children)[0]->set_width_chars(15);

        $self->{videoLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{video});
        $self->{video} = Gtk2::Combo->new;
		($self->{video}->get_children)[0]->set_width_chars(15);

        $self->{trailerLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{trailer});
        $self->{trailer} = new Gtk2::Entry;
        $self->{trailer}->signal_connect('changed' => sub {
            my $entry = shift;
            $self->{parent}->enablePlay($entry->get_text() ? 1 : 0);
        });
        $self->{trailerOpen} = Gtk2::Button->new_from_stock('gtk-open');
        $self->{trailerOpen}->signal_connect('clicked' => sub {
            $self->changeTrailer;
        });

        $self->{trailerExtract} = Gtk2::Button->new($parent->{lang}->{ExtractButton});
        $self->{trailerExtract}->signal_connect('clicked' => sub {
            $self->{parent}->extractInfo($self->{trailer}->get_text, $self);
        });

        $self->{collectionLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{collection});
        $self->{collection} = Gtk2::Combo->new;
		($self->{collection}->get_children)[0]->set_width_chars(20);
        $self->{rankLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{rank});
        my $adjRank = Gtk2::Adjustment->new(0, 0, 1000, 1, 10, 10) ;
        $self->{rank} = Gtk2::SpinButton->new($adjRank, 0, 0);
        $self->{rank}->set_numeric(1);

        $self->{frameLang} = new Gtk2::Frame;
        $self->{expanderLang} = new Gtk2::Expander($parent->{lang}->{Panel}->{languages});
        $self->{expanderLang}->set_expanded($options->expandLang);
        $self->{expanderLang}->signal_connect('activate' => sub {
            $self->refreshExpanderLang(1);
        });
        $self->{frameLang}->set_border_width(0);
        $self->{frameLang}->set_shadow_type('etched-in');
        $self->{expanderLang}->add($self->{frameLang});
       
        my $hboxAudioSub = new Gtk2::HBox(0,0);

        $self->{vboxAudio} = new Gtk2::VBox(0,0);
        $self->{audioList} = new Gtk2::SimpleList($parent->{lang}->{Panel}->{audio} => 'text',
                                                  $parent->{lang}->{Panel}->{encoding} => 'text');
        for my $i(0..1)
        {
            $self->{audioList}->get_column($i)->set_resizable(1);
        }
        $self->{audioList}->set_column_editable(1, 1);
        my $scrollAudio = new Gtk2::ScrolledWindow;
        $scrollAudio->set_policy ('automatic', 'automatic');
        $scrollAudio->set_shadow_type('etched-in');
        $scrollAudio->set_size_request(-1, 120);
        $scrollAudio->add($self->{audioList});
        $self->{vboxAudio}->pack_start($scrollAudio, 1, 1, 2);

        $self->{audio} = Gtk2::Combo->new;
		($self->{audio}->get_children)[0]->set_width_chars(10);
        $self->{encoding} = Gtk2::Combo->new;
		($self->{encoding}->get_children)[0]->set_width_chars(10);
        my $audioAddButton = Gtk2::Button->new_from_stock('gtk-add');
        my $tmpWidget = $audioAddButton;
        $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
        ($tmpWidget->get_children)[1]->destroy;        
        $audioAddButton->signal_connect('clicked' => sub {
            my $audio = ($self->{audio}->get_children)[0]->get_text;
            my $encoding = ($self->{encoding}->get_children)[0]->get_text;
            push @{$self->{audioList}->{data}}, [$audio, $encoding];
        });
        my $audioRemoveButton = Gtk2::Button->new_from_stock('gtk-remove');
        $tmpWidget = $audioRemoveButton;;
        $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
        ($tmpWidget->get_children)[1]->destroy;
        $audioRemoveButton->signal_connect('clicked' => sub {
            my @idx = $self->{audioList}->get_selected_indices;
            my $selected = $idx[0];
            splice @{$self->{audioList}->{data}}, $selected, 1;
            $selected-- if ($selected >= scalar(@{$self->{audioList}->{data}}));
            $selected = 0 if $selected < 0 ;
            $self->{audioList}->select($selected);
        });
		my $hboxAudio = new Gtk2::HBox(0,0);
		$hboxAudio->pack_start($self->{audio}, 0, 0, 2);
		$hboxAudio->pack_start($self->{encoding}, 0, 0, 2);
		$hboxAudio->pack_start($audioAddButton, 0, 0, 2);
		$hboxAudio->pack_start($audioRemoveButton, 0, 0, 2);
		$self->{vboxAudio}->pack_start($hboxAudio, 1, 1, 2);

		$hboxAudioSub->pack_start($self->{vboxAudio}, 0, 0, 4);

        $self->{vboxSub} = new Gtk2::VBox(0,0);
        $self->{subList} = new Gtk2::SimpleList($parent->{lang}->{Panel}->{subt} => 'text');
        my $scrollSub = new Gtk2::ScrolledWindow;
        $scrollSub->set_policy ('automatic', 'automatic');
        $scrollSub->set_shadow_type('etched-in');
        $scrollSub->set_size_request(-1, 120);
        $scrollSub->add($self->{subList});
        $self->{vboxSub}->pack_start($scrollSub, 1, 1, 2);
        $self->{subt} = Gtk2::Combo->new;
		my @subChildren = $self->{subt}->get_children;
		$subChildren[0]->set_width_chars(12);
        my $subAddButton = Gtk2::Button->new_from_stock('gtk-add');
        $tmpWidget = $subAddButton;
        $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
        ($tmpWidget->get_children)[1]->destroy;
        $subAddButton->signal_connect('clicked' => sub {
            my $sub = ($self->{subt}->get_children)[0]->get_text;
            push @{$self->{subList}->{data}}, [$sub];
        });
        my $subRemoveButton = Gtk2::Button->new_from_stock('gtk-remove');
        $tmpWidget = $subRemoveButton;
        $tmpWidget = $tmpWidget->child while ! $tmpWidget->isa('Gtk2::HBox');
        ($tmpWidget->get_children)[1]->destroy;
        $subRemoveButton->signal_connect('clicked' => sub {
            my @idx = $self->{subList}->get_selected_indices;
            my $selected = $idx[0];
            splice @{$self->{subList}->{data}}, $selected, 1;
            $selected-- if ($selected >= scalar(@{$self->{subList}->{data}}));
            $selected = 0 if $selected < 0 ;
            $self->{subList}->select($selected);
        });
		my $hboxSub = new Gtk2::HBox(0,0);
		$hboxSub->pack_start($self->{subt}, 0, 0, 2);
		$hboxSub->pack_start($subAddButton, 0, 0, 2);
		$hboxSub->pack_start($subRemoveButton, 0, 0, 2);
		$self->{vboxSub}->pack_start($hboxSub, 1, 1, 2);

		$hboxAudioSub->pack_start($self->{vboxSub}, 0, 0, 4);

		$self->{frameLang}->add($hboxAudioSub);

        $self->{commentLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{comment});
        $self->{commentLabel}->set_alignment(0,0);
        $self->{comment} = new Gtk2::TextView;
        $self->{comment}->set_editable(1);
        $self->{comment}->set_wrap_mode('word');
        $self->{scrollComment} = new Gtk2::ScrolledWindow;
        $self->{scrollComment}->set_border_width(2);
        $self->{scrollComment}->set_shadow_type('in');
        $self->{scrollComment}->set_policy('automatic', 'automatic');
        #$self->{scrollComment}->set_size_request(-1,80);
        $self->{scrollComment}->add($self->{comment});

        my $frameInfo = new Gtk2::Frame();
        $frameInfo->set_label_align(1.0, 0.0);

        $self->{origLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{orig});
        $self->{orig} = Gtk2::Entry->new;

        my $vbox = new Gtk2::VBox(0, 0);

        my $hboxpicinfos = new Gtk2::HBox(0,0);
        $self->{image} = new GCMovieImage($options, $self->{defaultImage});
        $self->{imgButton} = Gtk2::Button->new;
        $self->{imgButton}->add($self->{image});
        $self->{imgButton}->set_size_request(130,170);
        $self->{tooltips}->set_tip($self->{imgButton}, $parent->{lang}->{PanelImageTip});
        $self->{imgContext} = new Gtk2::Menu;
        $self->{imgContext}->append(Gtk2::TearoffMenuItem->new());
        $self->{imgOpen} = Gtk2::ImageMenuItem->new_from_stock('gtk-open',undef);
        $self->{imgOpen}->signal_connect("activate" , sub {
            $self->changeImage;
        });
        $self->{imgContext}->append($self->{imgOpen});
        my $itemShow = Gtk2::ImageMenuItem->new_from_stock('gtk-zoom-100',undef);
        $itemShow->signal_connect("activate" , sub {
            $self->showImage;
        });
        $self->{imgContext}->append($itemShow);
        $self->{imgClear} = Gtk2::ImageMenuItem->new_from_stock('gtk-clear',undef);
        $self->{imgClear}->signal_connect("activate" , sub {
            $self->clearImage;
        });
        $self->{imgContext}->append($self->{imgClear});
        $self->{imgContext}->show_all;
        $self->{imgButton}->signal_connect('clicked' => sub {
            $self->changeImage
        });
        $self->{imgButton}->signal_connect('button_press_event' => sub {
            my ($widget, $event) = @_;
            #return 0 if $self->{locked} || ($event->button ne 3);
            return 0 if ($event->button ne 3);
            $self->{imgContext}->popup(undef, undef, undef, undef, $event->button, $event->time);
            return 0;
        });

        $self->{imgButton}->drag_dest_set('all', ['copy','private','default','move','link','ask']);
        my $target_list = Gtk2::TargetList->new();
        my $atom1 = Gtk2::Gdk::Atom->new('text/uri-list');
        $target_list->add($atom1, 0, 0);
        if ($^O =~ /win32/i)
        {
            my $atom2 = Gtk2::Gdk::Atom->new('DROPFILES_DND');
            $target_list->add($atom2, 0, 0);
        }
        $self->{imgButton}->drag_dest_set_target_list($target_list);
        $self->{imgButton}->signal_connect(drag_data_received => sub {
            my ($widget, $context, $widget_x, $widget_y, $data, $info,$time) = @_;        
            my @files = split /\n/, $data->data;
            (my $fileName = $files[0]) =~ s/^file:\/\/\/?(.*)\W*$/$1/;
            $fileName =~ s/.$//ms;
            $self->changeImage($fileName);
        });

        $self->{infosFrame} = new Gtk2::Frame($parent->{lang}->{PanelInfoFrame});

        $self->{dateLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{date});
        $self->{dateLabel}->set_alignment(0,0.5);
        $self->{date} = Gtk2::Entry->new;

        $self->{directorLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{director});
        $self->{directorLabel}->set_alignment(0,0.5);
        $self->{director} = Gtk2::Entry->new;
        
        $self->{timeLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{time});
        $self->{timeLabel}->set_alignment(0,0.5);
        $self->{time} = Gtk2::Entry->new;

        $self->{natLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{nat});
        $self->{natLabel}->set_alignment(0,0.5);
        $self->{nat} = Gtk2::Entry->new;
        
        $self->{ageLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{age});
        $self->{ageLabel}->set_alignment(0,0.5);
        $self->{age} = new GCAgeChooser($parent->{lang}, 'minimum',100);

        my $vboxButton = new Gtk2::VBox(0,0);
        $vboxButton->pack_start($self->{imgButton},0,0,2);
        $hboxpicinfos->pack_start($vboxButton,0,0,2);
        $hboxpicinfos->pack_start($self->{infosFrame},0,0,20);

        $self->{hboxTypes} = new Gtk2::HBox(0,0);
        $self->{typeLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{type});
        $self->{type}[0] = Gtk2::Combo->new;
        $self->{type}[1] = Gtk2::Combo->new;
        $self->{type}[2] = Gtk2::Combo->new;
		my @children = $self->{type}[0]->get_children;
		$children[0]->set_width_chars(15);
		@children = $self->{type}[1]->get_children;
		$children[0]->set_width_chars(15);
		@children = $self->{type}[2]->get_children;
		$children[0]->set_width_chars(15);
        $self->{hboxTypes}->pack_start($self->{type}[0],0,0,2);
        $self->{hboxTypes}->pack_start($self->{type}[1],0,0,2);
        $self->{hboxTypes}->pack_start($self->{type}[2],0,0,2);

        $self->{actorsLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{actors});
        $self->{actors} = Gtk2::Entry->new;

        $self->{synopsisLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{synopsis});
        $self->{synopsis} = new Gtk2::TextView;
        $self->{synopsis}->set_editable(1);
        $self->{synopsis}->set_wrap_mode('word');
        $self->{scrollSynopsis} = new Gtk2::ScrolledWindow;
        $self->{scrollSynopsis}->set_border_width(2);
        $self->{scrollSynopsis}->set_shadow_type('in');
        $self->{scrollSynopsis}->set_policy('automatic', 'automatic');
        $self->{scrollSynopsis}->add($self->{synopsis});

        $self->{urlButton} = Gtk2::Button->new;
        if ($self->{parent}->{hasPluginsPictures})
        {
            $self->{urlImage} = new Gtk2::Image;
            $self->{urlButton}->add($self->{urlImage});
        }
        else
        {
            my $urlImage = Gtk2::Image->new_from_stock('gtk-media-next', 'button');
            my $urlBox = new Gtk2::HBox(0,0);
            my $urlBox2 = new Gtk2::HBox(0,0);
            $self->{urlLabel} = new Gtk2::Label;
            $urlBox2->pack_start($urlImage,0,0,5);
            $urlBox2->pack_start($self->{urlLabel},0,0,5);
            $urlBox->pack_start($urlBox2,1,0,0);
            $self->{urlButton}->add($urlBox);
        }
        
        $self->{urlButton}->signal_connect("clicked" => \&viewMoviePage, $self );
        $self->{tooltips}->set_tip($self->{urlButton}, $parent->{lang}->{PanelWebTip}, '');
        my $deleteButton = Gtk2::Button->new_from_stock('gtk-delete');
        $deleteButton->signal_connect("clicked" => \&deleteMovie, $self);
        $self->{tooltips}->set_tip($deleteButton, $parent->{lang}->{PanelRemoveTip}, '');

        $frameInfo->add($vbox);
        $frameInfo->set_border_width(2);

        $self->{mainBox}->pack_start($hboxTitle,0,0,2);
        
        $self->{frameLend} = new Gtk2::Frame();
        $self->{frameLend}->set_border_width(2);
        $self->{frameLend}->set_label_align(1.0, 0.0);
        my $vboxLend = new Gtk2::VBox(0,0);
        $vboxLend->set_border_width(2);
        
        $self->{borrowerLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{borrower});
        $self->{borrower} = new Gtk2::OptionMenu();
        
        $self->{lendDate} = Gtk2::Entry->new_with_max_length(10);
        $self->{lendDate}->set_width_chars(12);
        $self->{lendDateLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{lendDate});
        $self->{dateSelectButton} = new Gtk2::Button($parent->{lang}->{PanelLendDateSelect});
        $self->{dateSelectButton}->signal_connect('clicked' => sub {
                my $dialog = new GCDateSelectionDialog($parent);
                $dialog->date($self->{lendDate}->get_text);
                if ($dialog->show)
                {
                    $self->{lendDate}->set_text($dialog->date);
                }
                $self->showMe;
            });
            
        $self->{mailButton} = new Gtk2::Button($parent->{lang}->{MailTitle});
        $self->{mailButton}->signal_connect('clicked' => \&sendEmail, $self);
        $self->{returnedButton} = new Gtk2::Button($parent->{lang}->{PanelReturned});
        $self->{returnedButton}->signal_connect('clicked' => \&returnedMovie, $self);
        
        $self->{historyLabel} = new Gtk2::Label($parent->{lang}->{Panel}->{history});
        $self->{history} = new Gtk2::SimpleList($parent->{lang}->{Panel}->{borrower} => "text",
                                                $parent->{lang}->{Panel}->{lendDate} => "text",
                                                $parent->{lang}->{Panel}->{returnDate} => "text");
        $self->{history}->set_rules_hint(1);
        my $scrollPanelList = new Gtk2::ScrolledWindow;
        $scrollPanelList->set_policy ('never', 'automatic');
        $scrollPanelList->set_shadow_type('etched-in');
        $scrollPanelList->set_border_width(0);
        $scrollPanelList->add($self->{history});
        $self->{hboxHistory} = new Gtk2::HBox(0,0);
        $self->{hboxHistory}->pack_start($scrollPanelList,1,0,100);
        $self->{hboxHistoryActions} = new Gtk2::HBox(0,0);
        my $removeHistoryButton = new Gtk2::Button($parent->{lang}->{PanelRemove});
        $removeHistoryButton->signal_connect('clicked' => sub {
                my @idx = $self->{history}->get_selected_indices;
                if ($^O =~ /win32/i)
                {
                    my @newData;
                    my $i = 0;
                    foreach (@{$self->{history}->{data}})
                    {
                        push @newData, [$_->[0], $_->[1]] if $i != $idx[0];
                        $i++;
                    }
                    @{$self->{history}->{data}} = @newData;
                }
                else
                {
                    splice @{$self->{history}->{data}}, $idx[0], 1;
                }
            });
        my $clearHistoryButton = new Gtk2::Button($parent->{lang}->{PanelClear});
        $clearHistoryButton->signal_connect('clicked' => sub {
                @{$self->{history}->{data}} = ();
            });
        $self->{hboxHistoryActions}->pack_start($removeHistoryButton,0,0,10);
        $self->{hboxHistoryActions}->pack_start($clearHistoryButton,0,0,10);
        my $hboxLayout = new Gtk2::HBox(0,0);
        $hboxLayout->pack_start($self->{hboxHistoryActions},1,0,0);
 
        $self->{frameLend}->add($vboxLend);

        $self->deactivate if ($readonly ||  $parent->{options}->lockPanel) ;
        
        $self->{origLabel}->set_alignment(0,0.5);
        $self->{placeLabel}->set_alignment(0,0.5);
        $self->{videoLabel}->set_alignment(0,0.5);
        $self->{trailerLabel}->set_alignment(0,0.5);
        $self->{collectionLabel}->set_alignment(0,0.5);
        $self->{rankLabel}->set_alignment(0,0.5);
        $self->{typeLabel}->set_alignment(0,0.5);
        $self->{ratingLabel}->set_alignment(0,0.5);
        $self->{formatLabel}->set_alignment(0,0.5);
        $self->{numberLabel}->set_alignment(0,0.5);
        $self->{borrowerLabel}->set_alignment(0,0.5);
        $self->{lendDateLabel}->set_alignment(0,0.5);
        $self->{actorsLabel}->set_alignment(0,0.5);
        $self->{synopsisLabel}->set_alignment(0,0.5);        
        
        if ($options->order == 2)
        {
            #Compact layout
            $hboxTitle->pack_start($deleteButton,0,0,12);
        
            $hboxpicinfos->set_child_packing ($self->{infosFrame}, 1, 1, 10, 'start');
            $vbox->pack_start($hboxpicinfos,0,0,2);
            $vbox->set_border_width(2);
        
            $self->{infosFrame}->set_label(undef);
            $self->{infosFrame}->set_shadow_type('none');
            $frameInfo->set_shadow_type('none');
            $framePerso->set_shadow_type('none');
            $framePerso->set_border_width(0);
            $self->{frameLend}->set_label($parent->{lang}->{PanelLendInfo});
            $self->{frameLend}->set_shadow_type('etched-in');
            $self->{frameLend}->set_border_width(12);
            
            ($self->{place}->get_children)[0]->set_width_chars(8);
            ($self->{video}->get_children)[0]->set_width_chars(8);
            ($self->{collection}->get_children)[0]->set_width_chars(8);
            $frameInfo->set_border_width(0);

            $self->{date}->set_width_chars(18);
            $self->{director}->set_width_chars(18);
            $self->{nat}->set_width_chars(18);
            $self->{time}->set_width_chars(8);
            $self->{rating}->set_width_chars(3);
            $self->{rank}->set_width_chars(4);
            $self->{number}->set_width_chars(3);
            
            $self->{frameLend}->set_label_align(0.0, 0.0);

            $self->{scrollSynopsis}->set_size_request(-1,80);
            $self->{scrollComment}->set_size_request(-1,80);
            $self->{urlButton}->set_size_request(200,-1);

            my $infoTable = new Gtk2::Table(8, 8, 0);
            $infoTable->set_row_spacings(10);
            $infoTable->set_row_spacing(3,10);
            $infoTable->set_col_spacings(0);
            $infoTable->set_col_spacing(2,30);
            $infoTable->set_col_spacing(5,30);

            $infoTable->attach($self->{origLabel}, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{orig}, 1, 6, 0, 1, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{natLabel}, 6, 7, 0, 1, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{nat}, 7, 8, 0, 1, 'fill', 'fill', 0, 0);

            $infoTable->attach($self->{dateLabel}, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{date}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{directorLabel}, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{director}, 1, 2, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{collectionLabel}, 0, 1, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{collection}, 1, 2, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{formatLabel}, 0, 1, 4, 5, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{format}, 1, 2, 4, 5, 'fill', 'fill', 0, 0);

            $infoTable->attach($self->{timeLabel}, 3, 4, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{time}, 4, 5, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{ratingLabel}, 3, 4, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{rating}, 4, 5, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{rankLabel}, 3, 4, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{rank}, 4, 5, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{numberLabel}, 3, 4, 4, 5, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{number}, 4, 5, 4, 5, 'fill', 'fill', 0, 0);
            
            $infoTable->attach($self->{ageLabel}, 6, 7, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{age}, 7, 8, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{videoLabel}, 6, 7, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{video}, 7, 8, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{placeLabel}, 6, 7, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{place}, 7, 8, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{seen}, 6, 8, 4, 5, 'fill', 'fill', 0, 0);

            $infoTable->attach($self->{trailerLabel}, 0, 1, 5, 6, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{trailer}, 1, 5, 5, 6, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{trailerOpen}, 6, 7, 5, 6, 'fill', 'shrink', 0, 0);
            $infoTable->attach($self->{trailerExtract}, 7, 8, 5, 6, 'fill', 'fill', 5, 0);
            
            $infoTable->attach($self->{typeLabel}, 0, 1, 6, 7, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{hboxTypes}, 1, 8, 6, 7, 'fill', 'fill', 0, 0);

            $self->{fillLabel} = new Gtk2::Label;
            $self->{fillLabel}->set_markup("<span size='x-small'> </span>");
            $infoTable->attach($self->{fillLabel}, 0, 8, 7, 8, 'expand', 'fill', 0, 0);
            $self->{fillLabel}->set_size_request(-1,1);

            #$self->{expanderLang}->set_expanded(0);

            my $vboxInfo = new Gtk2::VBox();
            $vboxInfo->pack_start($infoTable, 0, 0, 0);
            $vboxInfo->pack_start($self->{expanderLang}, 1, 1, 0);

            $self->{infosFrame}->add($vboxInfo);
            
            my $lendTable = new Gtk2::Table(2,4,0);
            $lendTable->set_col_spacings(10);
            $lendTable->set_row_spacings(10);
            $lendTable->set_border_width(2);
            $lendTable->attach($self->{borrowerLabel}, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{borrower}, 1, 2, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{mailButton}, 2, 3, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{lendDateLabel}, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{lendDate}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{dateSelectButton}, 2, 3, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{returnedButton}, 3, 4, 1, 2, 'fill', 'fill', 0, 0);
            $vboxLend->pack_start($lendTable, 0, 0, 0);
            $vboxLend->pack_start($self->{historyLabel}, 0, 0, 0);
            $vboxLend->pack_start($self->{hboxHistory}, 1, 1, 5);
            $vboxLend->pack_start($hboxLayout, 1, 0, 5) if ! $readonly;

            $self->{vboxLongInfos} = new Gtk2::VBox(0,0);
            $self->{vboxLongInfos}->set_border_width(0);
            
            $self->{vboxLongInfos}->pack_start($self->{actorsLabel},0,0,0);
            $self->{vboxLongInfos}->pack_start($self->{actors},0,0,2);    
            
            $self->{vboxLongInfos}->pack_start($self->{synopsisLabel},0,0,0);
            $self->{vboxLongInfos}->pack_start($self->{scrollSynopsis},1,1,2);

            $self->{vboxLongInfos}->pack_start($self->{commentLabel},0,0,0);
            $self->{vboxLongInfos}->pack_start($self->{scrollComment},1,1,2);

            my $hboxLendUrl = new Gtk2::HBox(0,0);
            $hboxLendUrl->pack_start($self->{frameLend},1,1,0);
            my $vboxUrl = new Gtk2::VBox(0,0);
            $vboxUrl->pack_end($self->{urlButton},0,0,12);
            $hboxLendUrl->pack_end($vboxUrl,0,0,10);
            
            $self->{hboxLongInfos} = new Gtk2::HBox(0,0);
            $self->{vboxLongInfos}->set_border_width(0);
            $self->{hboxLongInfos}->pack_start($self->{vboxLongInfos},1,1,10);
            
            $self->{mainBox}->pack_start($frameInfo, 0, 0, 2);
            $self->{mainBox}->pack_start($self->{hboxLongInfos},1,1,2);
            $self->{mainBox}->pack_start($hboxLendUrl, 0, 0, 2);            
        }
        else
        {
            $self->{hboxOrig} = new Gtk2::HBox(0,0);
            $self->{hboxOrig}->pack_start($self->{origLabel},0,0,0);
            $self->{hboxOrig}->pack_start($self->{orig},1,1,0);

            $self->{frameAction} = new Gtk2::Frame();
            $self->{frameAction}->set_border_width(0);
            $self->{frameAction}->set_shadow_type('none');
            $self->{frameAction}->set_label_align(1.0, 0.0);
            my $hboxActions = new Gtk2::HBox(1,0);
            $hboxActions->set_border_width(0);
            $hboxActions->pack_start($self->{urlButton},1,1,20);
            $hboxActions->pack_start($deleteButton,1,1,20);
            $self->{frameAction}->add($hboxActions);
        
            $self->{hboxActors} = new Gtk2::HBox(0,0);
            $self->{hboxActors}->pack_start($self->{actorsLabel},0,0,0);
            $self->{hboxActors}->pack_start($self->{actors},1,1,0);    
        
            $self->{hboxSynopsis} = new Gtk2::HBox(0,0);
            $self->{hboxSynopsis}->pack_start($self->{synopsisLabel},0,0,0);
            $self->{hboxSynopsis}->pack_start($self->{scrollSynopsis},1,1,0);        

            $self->{hboxTypes}->pack_start($self->{typeLabel},0,0,0);
            $self->{hboxTypes}->reorder_child($self->{typeLabel},0);

            $vbox->pack_start($self->{hboxOrig},0,0,2);
            $vbox->pack_start($hboxpicinfos,0,0,2);
            $vbox->pack_start($self->{hboxTypes},0,0,10);
            $vbox->pack_start($self->{hboxActors},0,0,2);
            $vbox->pack_start($self->{hboxSynopsis},1,1,2);
            $vbox->set_border_width(2);
                
            my $infoTable = new Gtk2::Table(5, 2, 0);
            $infoTable->set_row_spacings(5);
            $infoTable->set_border_width(5);
            $infoTable->attach($self->{dateLabel}, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{date}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{directorLabel}, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{director}, 1, 2, 2, 3, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{timeLabel}, 0, 1, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{time}, 1, 2, 3, 4, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{natLabel}, 0, 1, 4, 5, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{nat}, 1, 2, 4, 5, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{ageLabel}, 0, 1, 5, 6, 'fill', 'fill', 0, 0);
            $infoTable->attach($self->{age}, 1, 2, 5, 6, 'fill', 'fill', 0, 0);
            $self->{infosFrame}->add($infoTable);

            my $detailsTable = new Gtk2::Table(5, 6, 0);
            $detailsTable->set_row_spacings(10);
            $detailsTable->set_col_spacings(10);
            $detailsTable->attach($self->{seen}, 0, 2, 0, 1, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{ratingLabel}, 2, 3, 0, 1, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{rating}, 3, 4, 0, 1, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{formatLabel}, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{format}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{numberLabel}, 2, 3, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{number}, 3, 4, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{placeLabel}, 4, 5, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{place}, 5, 6, 1, 2, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{videoLabel}, 0, 1, 2, 3, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{video}, 1, 4, 2, 3, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{collectionLabel}, 0, 1, 3, 4, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{collection}, 1, 4, 3, 4, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{rankLabel}, 4, 5, 3, 4, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{rank}, 5, 6, 3, 4, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{trailerLabel}, 0, 1, 4, 5, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{trailer}, 1, 4, 4, 5, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{trailerOpen}, 4, 5, 4, 5, 'fill', 'fill', 0, 0);
            $detailsTable->attach($self->{trailerExtract}, 5, 6, 4, 5, 'fill', 'fill', 0, 0);

            $self->{vboxComment} = new Gtk2::VBox(0,0);
            $self->{vboxComment}->set_border_width(0);
            $self->{vboxComment}->pack_start($self->{commentLabel},0,0,0);
            $self->{vboxComment}->pack_start($self->{scrollComment},1,1,0);

            $vboxPerso->pack_start($detailsTable,0,0,0);
            $vboxPerso->pack_start($self->{expanderLang},0,1,2);
            $vboxPerso->pack_start($self->{vboxComment},1,1,2);
            $framePerso->add($vboxPerso);

            my $lendTable = new Gtk2::Table(2,4,0);
            $lendTable->set_col_spacings(10);
            $lendTable->set_row_spacings(10);
            $lendTable->set_border_width(2);
            $lendTable->attach($self->{borrowerLabel}, 0, 1, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{borrower}, 1, 2, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{mailButton}, 2, 3, 0, 1, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{lendDateLabel}, 0, 1, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{lendDate}, 1, 2, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{dateSelectButton}, 2, 3, 1, 2, 'fill', 'fill', 0, 0);
            $lendTable->attach($self->{returnedButton}, 3, 4, 1, 2, 'fill', 'fill', 0, 0);
            $vboxLend->pack_start($lendTable, 0, 0, 0);
            my $spacerBox = new Gtk2::VBox();
            $spacerBox->set_size_request(-1,15);
            $vboxLend->pack_start($spacerBox, 0, 0, 0);
            $vboxLend->pack_start($self->{historyLabel}, 0, 0, 5);
            $vboxLend->pack_start($self->{hboxHistory}, 1, 1, 5);
            $vboxLend->pack_start($hboxLayout, 0, 0, 5) if ! $readonly;

            if ($options->order == 0)
            {
                #Tabs layout
                $frameInfo->set_shadow_type('none');
                $framePerso->set_shadow_type('none');
                $self->{frameLend}->set_shadow_type('none');
                $self->{tabs} = Gtk2::Notebook->new();
                
                $self->{tabs}->set_name('GCInfoTabs');
                
                $self->{tabs}->append_page($frameInfo, $parent->{lang}->{PanelMovieInfo});
                $self->{tabs}->append_page($framePerso, $parent->{lang}->{PanelPrivateInfo});
                $self->{tabs}->append_page($self->{frameLend}, $parent->{lang}->{PanelLendInfo});
                
                $self->{mainBox}->pack_start($self->{tabs}, 1, 1, 2);
            }
            else
            {
                #All In One layout
                $frameInfo->set_label($parent->{lang}->{PanelMovieInfo});
                $framePerso->set_label($parent->{lang}->{PanelPrivateInfo});
                $self->{frameLend}->set_label($parent->{lang}->{PanelLendInfo});
                $frameInfo->set_shadow_type('etched-in');
                $framePerso->set_shadow_type('etched-in');
                $self->{frameLend}->set_shadow_type('etched-in');
                $self->{mainBox}->pack_start($frameInfo, 1, 1, 2);
                $self->{mainBox}->pack_start($framePerso, 0, 0, 2);
                $self->{mainBox}->pack_start($self->{frameLend}, 1, 1, 2);
            }
            
            $self->{mainBox}->pack_start($self->{frameAction}, 0, 0, 2) if ! $readonly;
        }

        $self->set_name('GCMovieFrame');

        my $realMain = new Gtk2::VBox(0,0);

        $realMain->pack_start($self->{mainBox},1,1,0);

        $self->add($realMain);
        $self->show_all;

        $self->{warning} = new Gtk2::Label;
        
        $self->{infoBox}->pack_start($self->{warning},1,1,0);
        $self->{infoBox}->pack_start($self->{viewAllBox},1,1,0);
        $realMain->pack_start($self->{infoBox},1,1,0);
        
        return $self;
    }

    sub searchMovie
    {
        my ($widget, $self) = @_;

        $self->{parent}->searchMovieForPanel($self->title,$self);

        $self->showMe;
    }

    sub checkBorrowerButtons
    {
        my $self = shift;
        $self->{mailButton}->set_sensitive($self->{hasBorrower} && !$self->{locked});
        $self->{returnedButton}->set_sensitive($self->{hasBorrower} && !$self->{locked});
    }
    
    sub deactivate
	{
        my $self = shift;
		$self->changeState($self,1);
	}

	sub changeState
    {
        my ($caller, $self, $locked)  = @_;
        
        $self->{locked} = $locked;
        
        my $i = 0;
        foreach (@GCMovies::fieldsArray)
        {
            $i++;
            next if !($i % 2);
            next if ($_ eq 'type')
                 || ($_ eq 'image')
                 || ($_ eq 'url')
                 || ($_ eq 'audio')
                 || ($_ eq 'subt');
            
            if ($self->{$_}->isa('Gtk2::SpinButton'))
            {
                $self->{$_}->can_focus(!$locked);
                my $step = ($locked ? 0 : 1);
                $self->{$_}->set_increments($step, $step);
            }
            elsif (($self->{$_}->isa('Gtk2::Entry'))
                || ($self->{$_}->isa('Gtk2::TextView')))
            {
                $self->{$_}->can_focus(!$locked);
            }
            elsif ($self->{$_}->isa('Gtk2::Combo'))
            {
                ($self->{$_}->get_children)[0]->can_focus(!$locked);
                ($self->{$_}->get_children)[1]->set_sensitive(!$locked);
            }
            else
            {
                $self->{$_}->set_sensitive(!$locked);
            }
        }
		
		for my $button('search', 'dateSelect')
		{
		  $self->{$button.'Button'}->set_sensitive(!$locked);
		  $self->{$button.'Button'}->hide;
		  $self->{$button.'Button'}->show;
		}
		
		for my $menu('Open', 'Clear')
		{
		  $self->{'img'.$menu}->set_sensitive(!$locked);
		}		
		
		$self->checkBorrowerButtons;

        $self->{vboxAudio}->set_sensitive(!$locked);
        $self->{vboxSub}->set_sensitive(!$locked);
        $self->{trailerOpen}->set_sensitive(!$locked);
        $self->{trailerExtract}->set_sensitive(!$locked);
		
        $self->{type}[0]->can_focus(!$locked);
        $self->{type}[1]->can_focus(!$locked);
        $self->{type}[2]->can_focus(!$locked);
        my @children;
        @children = $self->{type}[0]->get_children;
		$children[0]->can_focus(!$locked);
		$children[1]->set_sensitive(!$locked);
        @children = $self->{type}[1]->get_children;
		$children[0]->can_focus(!$locked);
		$children[1]->set_sensitive(!$locked);
        @children = $self->{type}[2]->get_children;
		$children[0]->can_focus(!$locked);
		$children[1]->set_sensitive(!$locked);
    }
    
    sub setShowOption
    {
        my ($self, $show, $hasToShow) = @_;
       
        return if ! $self->initShow($show, $hasToShow);
        
        foreach (keys %{$show})
        {
            $self->{$_.'Label'}->hide if (! $show->{$_}) && ($self->{$_.'Label'});
            next if ($_ eq 'type')
                 || ($_ eq 'history');
            $self->{$_}->hide if (! $show->{$_}) && ($self->{$_});
        }
        
        $self->{trailerOpen}->hide if ! $show->{trailer};
        $self->{trailerExtract}->hide if ! $show->{trailer};
        $self->{searchButton}->hide if ! $show->{search};
        $self->{frameAction}->hide if ! $show->{actions} && $self->{frameAction};
        $self->{urlButton}->hide if ! $show->{actions} && ! $self->{frameAction};
        $self->{vboxAudio}->hide if ! $show->{audio};
        $self->{vboxSub}->hide if ! $show->{subt};
        $self->{expanderLang}->hide if (! $show->{subt}) && (! $show->{audio});
        if (! $show->{comment})
        {
            $self->{vboxComment}->hide if ($self->{vboxComment});
            $self->{scrollComment}->hide if $self->{scrollComment};
        }
        $self->{hboxOrig}->hide if ((! $show->{orig}) && $self->{hboxOrig});
        $self->{imgButton}->hide if ! $show->{image};
        $self->{infosFrame}->hide if ($self->{options}->order != 2)
                                  && (! $show->{date})
                                  && (! $show->{director})
                                  && (! $show->{time})
                                  && (! $show->{nat})
                                  && (! $show->{age});
        $self->{hboxTypes}->hide if ! $show->{type};
        $self->{hboxActors}->hide if ! $show->{actors} && $self->{hboxActors};
        if (! $show->{synopsis})
        {
            $self->{hboxSynopsis}->hide if $self->{hboxSynopsis};
            $self->{scrollSynopsis}->hide if $self->{scrollSynopsis};
        }
        $self->{hboxBorrower}->hide if ((! $show->{borrower}) && $self->{hboxBorrower});
        if (! $show->{lendDate})
        {
            if ($self->{hboxLendDate})
            {
                $self->{hboxLendDate}->hide;
            }
            else
            {
                $self->{dateSelectButton}->hide;
            }
        }
        $self->{mailButton}->hide if ! $show->{mail};
        $self->{hboxHistory}->hide if ! $show->{history};
        $self->{hboxHistoryActions}->hide if ! $show->{history};
        $self->{returnedButton}->hide if ! $show->{history};

        $self->{frameLend}->hide if $self->{frameLend}->get_label
                                && ! $show->{borrower}
                                && ! $show->{lendDate}
                                && ! $show->{mail}
                                && ! $show->{history};

        $self->{vboxLongInfos}->hide if ($self->{vboxLongInfos})
                                    && ! $show->{actors}
                                    && ! $show->{synopsis}
                                    && ! $show->{comment};

   }

   sub returnedMovie
   {
        use GCDialogs;

        shift;
        my $self = shift;

        my $dialog = new GCDateSelectionDialog($self->{parent});
        if ($dialog->show)
        {
            my @data = [$self->borrower, $self->lendDate, $dialog->date];
            unshift @{$self->{history}->{data}}, @data;
            my $previous = $self->getAsHash;
            $self->borrower('none');
            $self->lendDate('');
            my $new = $self->getAsHash;           
            $self->{parent}->{moviesList}->changeCurrent($previous, $new) if $self->image;
        }
        $self->showMe;
   }

    sub sendEmail
    {
        use GCDialogs;

        shift;
        my $self = shift;

        my %info;

        $info{title} = $self->title;
        $info{borrower} = $self->borrower;
        $info{lendDate} = $self->lendDate;

        my $dialog = new GCMailDialog($self->{parent}, \%info);

        $dialog->show;
        $self->showMe;
    }

    sub selectTitle
    {
        my $self = shift;
        $self->{title}->select_region(0, length($self->{title}->get_text()));
        $self->{title}->grab_focus;
    }

    sub deleteMovie
    {
       my ($widget, $self) = @_;

       $self->{parent}->deleteCurrentMovie;
    }

    sub setBrowser
    {
        my ($self, $cmd) = @_;

        $self->{browser} = $cmd;
    }

    sub viewMoviePage
    {
        my ($widget, $self) = @_;
        $self->{parent}->openUrl($self->url);
    }

    sub clearImage
    {
		my $self = shift;

        my $previous = $self->getAsHash;
        my $imagePrefix = $self->{parent}->{imagePrefix};
        if ($self->{imageFile} =~ /(\/|\\)$imagePrefix[0-9]*\./)
        {
            $self->{parent}->{movies}->markToBeRemoved($self->{imageFile});
        }
        $self->image('');
        my $new = $self->getAsHash;
        $self->{parent}->{moviesList}->changeCurrent($previous, $new);
    }

    sub changeImage
    {
        my ($self, $fileName) = @_;

        return 0 if $self->{locked};

        if (!$fileName)
        {
            if (! $self->{imageDialog})
            {
                $self->{imageDialog} = GCFileChooserDialog->new($self->{parent}->{lang}->{PanelImageTitle}, $self->{parent}, 'open');
                $self->{imageDialog}->setWithImagePreview(1);    
            }

            $self->{imageDialog}->set_filename($self->{imageFile});
            my $response = $self->{imageDialog}->run;
            $self->{imageDialog}->hide;
            $self->showMe;
            if ($response eq 'ok')
            {
                $fileName = $self->{imageDialog}->get_filename;
            }
            else
            {
                return;
            }
        }

        if ($fileName ne $self->{imageFile})
        {
            my $imagePrefix = $self->{parent}->{imagePrefix};
            if ($self->{imageFile} =~ /(\/|\\)$imagePrefix[0-9]*\./)
            {
                $self->{parent}->{movies}->markToBeRemoved($self->{imageFile});
            }
        }
        my $image = $self->{parent}->transformPicturePath($fileName);
        my $previous = $self->getAsHash;
        $self->image($image);
        my $new = $self->getAsHash;
        $self->{parent}->{moviesList}->changeCurrent($previous, $new);
    }
    
    sub showImage
    {
        my $self = shift;
        my $dialog = new GCImageDialog($self->{parent}, $self->{imageFile});
        $dialog->show;
        $dialog->destroy;
    }
    
    sub changeTrailer
    {
		my $self = shift;

        if (! $self->{trailerDialog})
        {
            $self->{trailerDialog} = GCFileChooserDialog->new($self->{parent}->{lang}->{PanelTrailerTitle}, $self->{parent}, 'open');
        }

		$self->{trailerDialog}->set_filename($self->{trailer}->get_text);
		my $response = $self->{trailerDialog}->run;
		if ($response eq 'ok')
		{
	        $self->{trailer}->set_text($self->{trailerDialog}->get_filename);
        }
        $self->{trailerDialog}->hide;
        $self->showMe;
    }

    sub history
    {
        my $self = shift;

        if (@_)
        {
            my $value = shift;
            @{$self->{history}->{data}} = ();
            my @lines = split m/,/, $value;
            foreach (@lines)
            {
                my @values = split m/:/;
                my @data = [ $values[0], $values[1], $values[2]];
                push @{$self->{history}->{data}}, @data;
            }
        }
        else
        {
            my $result = '';
            foreach (@{$self->{history}->{data}})
            {
                $result .= $_->[0].':'.$_->[1].':'.$_->[2].',';
            }
            $result =~ s/.$//;
            return $result;
        }
    }

    sub type
    {
        my $self = shift;
        if (@_)
        {
            $_ = shift;
            my @styles = split m/,/;
            my @stylesLists = @{$self->{type}};
            my $style;
            foreach $style (@stylesLists)
            {
                my @children = $style->get_children;
                my $type = shift @styles;
                $type =~ s/^\s*//;
                $children[0]->set_text($type);
            }
        }
        else
        {
            my $result = '';
            my @stylesLists = @{$self->{type}};
            my $style;
            foreach $style (@stylesLists)
            {
                my @children = $style->get_children;
                $result .= $children[0]->get_text if @children[0];
                $result .= ',';
            }
            return $result;
        }
    }

    sub place
    {
        my $self = shift;
        my @children = $self->{place}->get_children;
        if (@_)
        {
            my $value = shift;
            $children[0]->set_text($value);
        }
        else
        {
            return $children[0]->get_text if $children[0];
        }
    }

    sub video
    {
        my $self = shift;
        my @children = $self->{video}->get_children;
        if (@_)
        {
            my $value = shift;
            $children[0]->set_text($value);
        }
        else
        {
            return $children[0]->get_text if $children[0];
        }
    }

    sub audio
    {
        my $self = shift;
        if (@_)
        {
            $_ = shift;
            @{$self->{audioList}->{data}} = ();
            my @audios = split m/,/;

            foreach my $info(@audios)
            {
                my @track = split m/;/, $info;
                push @{$self->{audioList}->{data}}, \@track;
            }
            $self->refreshExpanderLang(0);
        }
        else
        {
            my $result = '';

            foreach (@{$self->{audioList}->{data}})
            {
                $result .= join ';', @$_;
                $result .= ',';
            }

            return $result;
        }
    }

    sub subt
    {
        my $self = shift;

        if (@_)
        {
            $_ = shift;
            @{$self->{subList}->{data}} = ();
            my @subs = split m/,/;

            foreach my $info(@subs)
            {
                push @{$self->{subList}->{data}}, [$info];
            }
            $self->refreshExpanderLang(0);
        }
        else
        {
            my $result = '';

            foreach (@{$self->{subList}->{data}})
            {
                $result .= $_->[0].',';
            }

            return $result;
        }
    }

    sub getLangsAsString
    {
        my $self = shift;

        my $audio = $self->audio;
        my $subt = $self->subt;

        $audio =~ s/;.*?(,|$)/$1/g;
        $audio =~ s/,$//;
        $subt =~ s/,$//;

        return '['.$self->{lang}->{Panel}->{audio}.$self->{lang}->{Separator}.$audio
          .'] - ['.$self->{lang}->{Panel}->{subt}.$self->{lang}->{Separator}.$subt.']';
    }

    sub refreshExpanderLang
    {
        my ($self, $beforeAction) = @_;
        if ($beforeAction xor ($self->{expanderLang}->get_expanded))
        {
            $self->{expanderLang}->set_label($self->{lang}->{PanelLanguages});
        }
        else
        {
            $self->{expanderLang}->set_label($self->{lang}->{PanelLanguages}.
                                             ' - '.$self->getLangsAsString);
        }
    }

    sub format
    {
        my $self = shift;
		if (@_)
		{
		    my $value = shift;
		    my $i = 0;
		    my $format;
		    foreach $format(@{$self->{formats}})
		    {
                last if (!$value) || ($format eq $value);
                $i++;
		    }
		    $i = 0 if $i >= scalar @{$self->{formats}};
		    $self->{formatMenu}->set_active($i);
		    $self->{format}->remove_menu;
		    $self->{format}->set_menu($self->{formatMenu});
		}
		else
		{
		    return $self->{format}->child->get_label if $self->{format}->child;
		}
    }

    sub borrower
    {
        my $self = shift;
		if (@_)
		{
		    my $value = shift;
		    my $i = 0;
            if ($value && ($value ne 'none'))
            {
                foreach (@{$self->{borrowers}})
                {
                    last if ($_ eq $value);
                    $i++;
                }
                $self->{hasBorrower} = 1;
            }
            else
            {
                $self->{hasBorrower} = 0;
            }

    		$self->checkBorrowerButtons;

            $i = scalar @{$self->{borrowers}} - 1 if $i >= scalar(@{$self->{borrowers}});
 		    $self->{borrowerMenu}->set_active($i);
		    $self->{borrower}->remove_menu;
		    $self->{borrower}->set_menu($self->{borrowerMenu});
		}
		else
		{
		    if ($self->{borrower}->child)
		    {
                return 'none' if $self->{borrower}->child->get_label eq $self->{lang}->{PanelNobody};
		        return $self->{borrower}->child->get_label;
		    }
		    else
		    {
		        return 'none' if $self->{borrowerMenu}->get_active->child->get_label eq $self->{lang}->{PanelNobody};
		        return $self->{borrowerMenu}->get_active->child->get_label;
		    }
		}
    }

    sub comment
    {
        my $self = shift;
        my $buffer = $self->{comment}->get_buffer;
        if (@_)
        {
            my $text = shift;
		    $text =~ s/<br>/\n/g;
		    $buffer->set_text($text);
        }
		else
		{
		    my $text = $buffer->get_text($buffer->get_start_iter,
						 $buffer->get_end_iter, 1);
		    $text =~ s/\n/<br>/g;
		    return $text;
		}
    }

    sub synopsis
    {
        my $self = shift;
        my $buffer = $self->{synopsis}->get_buffer;
        if (@_)
        {
            my $text = shift;
            $text =~s/<br>/\n/g;
            $buffer->set_text($text);
        }
		else
		{
		    my $text = $buffer->get_text($buffer->get_start_iter,
						 $buffer->get_end_iter, 1);
		    $text =~s/\n/<br>/g;
		    return $text;
		}
    }

    sub url
    {
        #use GCPlugins;
        my $self = shift;
        if (@_)
        {
            my $url = shift;
            $self->{url} = $url;
            if ($self->{urlType} eq 'text')
            {
                $self->{urlButton}->set_sensitive($url ? 1 : 0);
                return;   
            }
            my $baseUrl;
            my $plugin;
            foreach (values %GCPlugins::pluginsMap)
            {
                ($baseUrl = $_->getMovieUrl) =~ s/http:\/\/(w{3})?//;
                 $plugin = $_;
                last if $url =~ m/$baseUrl/;
            }
            if ($url =~ m/$baseUrl/)
            {
                if ($self->{parent}->{hasPluginsPictures})
                {
                    my $picFile = $self->{parent}->{pluginsPicturesDir}.$plugin->getImageFile;
                    $picFile = $ENV{GCF_SHARE_DIR}.'/icons/gcfilms_32x32.png' if ! -f $picFile;
                    $self->{urlImage}->set_from_file($picFile);
                    $self->{urlImage}->set_size_request(128,32);
                }
                else
                {
                    #$self->{urlButton}->set_label($plugin->getName);
                    $self->{urlLabel}->set_label($plugin->getName);
                }
                $self->{urlButton}->set_sensitive(1);
            }
            else
            {
                if ($self->{parent}->{hasPluginsPictures})
                {
                    $self->{urlImage}->set_from_file($self->{parent}->{pluginsPicturesDir}.'default.png');
                    $self->{urlImage}->set_size_request(128,32);
                }
                else
                {
                    #$self->{urlButton}->set_label('');
                    $self->{urlLabel}->set_label('');
                }
                $self->{urlButton}->set_sensitive(0);
            }
        }
        return $self->{url};
    }

   sub seen
   {
        my $self = shift;
        if (@_)
        {
            my $value = shift;
            $self->{seen}->set_active($value);
        }
        else
        {
            return 1 if ($self->{seen}->get_active);
            return 0;
        }
   }

    sub image
    {
        my $self = shift;

        if (@_)
        {
            $self->{imageFile} = shift;
            $self->{image}->setImage($self->{imageFile});
        }
        else
        {
            return $self->{imageFile};
        }
    }

    sub setBorrowers
    {
        my ($self, $values) = @_;

        my $current = $self->borrower if $self->{borrowerMenu};

        my @borrowers = ($self->{lang}->{PanelNobody}, @{$values}, $self->{lang}->{PanelUnknown});

        $self->{borrower}->remove_menu;
        $self->{borrowerMenu}->destroy if $self->{borrowerMenu};
        $self->{borrowerMenu} = new Gtk2::Menu();

        foreach (@borrowers)
        {
            my $item = Gtk2::MenuItem->new_with_mnemonic($_);
		    $self->{borrowerMenu}->append($item);
        }
        $self->{borrowers} = \@borrowers;
		$self->{borrower}->set_menu($self->{borrowerMenu});

        $self->borrower($current) if $self->{borrowerMenu};

        $self->{borrower}->show_all;

        $self->{borrowerMenu}->signal_connect(
          'selection-done' => sub {
            my $widget = shift;
            my $borrower;
            my $previous = $self->getAsHash;
            $previous->{borrower} = '0';
            return if !$widget->get_active->child;
            if ($widget->get_active->child->get_label eq $self->{lang}->{PanelNobody})
            {
                $borrower = 'none';
                $self->{hasBorrower} = 0;
            }
            else
            {
                $borrower = $widget->get_active->child->get_label;
                $self->{hasBorrower} = 1;
            }

    		$self->checkBorrowerButtons;
            my $new = $self->getAsHash;
            $new->{borrower} = $borrower;
            $self->{parent}->{moviesList}->changeCurrent($previous, $new)
                if $self->image && (! $self->{disableBorrowerChange});
        });
    }
    
    sub setFormats
    {
        my ($self, $values) = @_;

        my $current = $self->format if $self->{formatMenu};

        my @formats = @{$values};

        $self->{format}->remove_menu;
        $self->{formatMenu}->destroy if $self->{formatMenu};
        $self->{formatMenu} = new Gtk2::Menu();

        foreach (@formats)
        {
            my $item = Gtk2::MenuItem->new_with_mnemonic($_);
		    $self->{formatMenu}->append($item);
        }
        $self->{formats} = \@formats;
		$self->{format}->set_menu($self->{formatMenu});
        $self->format($current) if $self->{formatMenu};
        $self->{format}->show_all;
    }

    sub setHistory
    {
        my ($self, $field, $values) = (shift, shift, shift);

        if ($field eq 'type')
        {
            my @stylesLists = @{$self->{type}};
            my $style;
            foreach $style (@stylesLists)
            {
                $style->set_popdown_strings(@$values);
            }
            return;
        }
        elsif ($field eq 'audio')
        {
            my $encodings = shift;
            $self->{encoding}->set_popdown_strings(@$encodings);
        }
        
        $self->{$field}->set_popdown_strings(@$values);
    }

    sub AUTOLOAD
    {
        my $self = shift;
        my $name = our $AUTOLOAD;
        return if $name =~ /::DESTROY$/;
        $name =~ s/.*?::(.*)/$1/;
        
        
        if ($self->{parent}->{movies}->{fieldsInfo}->{$name}->{hasHistory})
        {
            #Combo boxes
            my @children = $self->{$name}->get_children;
            if (@_)
            {
                my $value = shift;
                $children[0]->set_text($value);
            }
            else
            {
                return $children[0]->get_text if $children[0];
            }
        }
        else
        {
            #Default for text fields
            if (scalar @_)
            {
                my $text = shift;
                if ($self->{parent}->{movies}->{fieldsInfo}->{$name}->{isNumeric})
                {
                    $self->{$name}->set_value($text);
                }
                else
                {
                    $self->{$name}->set_text($text);
                }
            }
            else
            {
                return $self->{$name}->get_text;
            }
        }
    }
}

1;
