/*
 * cowbell
 * Copyright (c) 2005 Todd Berman, Lluis Sanchez, Brad Taylor
 *
 * This file was derived from MonoDevelop.Services.DispatchService.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using GLib;

using System;
using System.Threading;
using System.Collections;

namespace Cowbell.Base
{
	public class Dispatcher
	{
		/* public delegates */
		public delegate void DelegateWrapper (Delegate d, object[] args);

		/* public fields */
		public DelegateWrapper wrapper;
		    
		/* public methods */
		public Dispatcher ()
		{
			gui_queue = new Queue ();
			wrapper = new DelegateWrapper (OnInvokeWrappedDelegate);
			handler = new GLib.IdleHandler (OnIdleHandler);
		}

		public void GuiDispatch (Delegate d, params object[] args)
		{
			lock (gui_queue) {
				gui_queue.Enqueue (new MethodCallWrapper (d, args));
				if (idle == 0) {
					idle = GLib.Idle.Add (handler);
				}
			}
		}

		public void BackgroundDispatch (Delegate d, params object[] args)
		{
			BackgroundDispatch (d, new AsyncCallback (OnEndWrapperInvoke), args);
		}
		
		public void BackgroundDispatch (Delegate d, AsyncCallback c, params object[] args)
		{
			wrapper.BeginInvoke (d, args, c, null);
		}
		
		/* private fields */
		private Queue gui_queue;
		private uint idle = 0;
		private IdleHandler handler;

		/* private methods */
		private void OnInvokeWrappedDelegate (Delegate d, object[] args)
		{
			d.DynamicInvoke (args);
		}

		private void OnEndWrapperInvoke (IAsyncResult ar)
		{
			wrapper.EndInvoke (ar);
		}

		private bool OnIdleHandler ()
		{
			lock (gui_queue) {
				if (gui_queue.Count == 0) {
					idle = 0;
					return false;
				}
			}

			MethodCallWrapper method = null;
			lock (gui_queue) {
				method = (MethodCallWrapper)gui_queue.Dequeue ();
			}

			if (method != null) {
				method.Invoke ();
			}

			if (gui_queue.Count == 0) {
				idle = 0;
				return false;
			}

			return true;
		}

		/* private classes */
		private class MethodCallWrapper
		{
			Delegate d;
			object[] args;
			
			public MethodCallWrapper (Delegate d, params object[] args)
			{
				this.d = d;
				this.args = args;
			}

			public void Invoke ()
			{
				d.Method.Invoke (d.Target, args);
			}
		}
	}
}
